import typing
import xbmc
import xbmcaddon
import xbmcgui
from resources.lib.exception import PluginException
from resources.lib.odysee import *
from resources.lib.local import get_user_channel
from resources.lib.comment_window import CommentWindow
from xbmcgui import Dialog, ListItem, WindowXMLDialog

tr = xbmcaddon.Addon().getLocalizedString

class NotificationWindowXML(WindowXMLDialog):
    def __init__(self, *args, **kwargs):
        WindowXMLDialog.__init__(self, *args, **kwargs)
        self.odysee : Odysee = kwargs['odysee']
        self.last_selected_position = -1
        self.has_notifications = False

    def onInit(self):
        try:
            self.refresh()
        except PluginException as e:
            raise e

    def onAction(self, action):
        try:
            self._onAction(action)
        except PluginException as e:
            Dialog().ok(tr(30100), str(e))
            xbmc.log("PluginException: " + str(e))
        except Exception as e:
            xbmc.log("Exception: " + str(e))

    def _unread_label_prefix(self) -> str:
        return '[COLOR orange]*[/COLOR] '

    def _remove_unread_prefix(self, li: xbmcgui.ListItem):
        label = li.getLabel()
        label = label.split(self._unread_label_prefix())[1]
        li.setLabel(label)
        li.setProperty('is_read', 'True')

    def _add_unread_prefix(self, li: xbmcgui.ListItem):
        label = li.getLabel()
        label = self._unread_label_prefix() + label
        li.setLabel(label)
        li.setProperty('is_read', 'False')

    def _onAction(self, action):
        if action == xbmcgui.ACTION_CONTEXT_MENU:
            ccl = self.get_notification_control_list()

            if not self.has_notifications:
                menu = []
                menu.append(tr(30240)) # Refresh
                ret = Dialog().contextmenu(menu)
                if ret == 0:
                    self.refresh()
                return

            item = ccl.getSelectedItem()
            if item:
                is_comment = item.getProperty('is_comment') == 'True'
                menu = []
                if is_comment:
                    menu.append(tr(30315)) # Open

                is_read = item.getProperty('is_read') == 'True'
                menu.append(tr(30316 if is_read else 30317)) # Mark as unread/read
                menu.append(tr(30318)) # Delete
                menu.append(tr(30319)) # Mark all as read
                menu.append(tr(30320)) # Mark all as unread
                menu.append(tr(30321)) # Delete all
                menu.append(tr(30240)) # Refresh

                ret = Dialog().contextmenu(menu)
                if ret == -1:
                    return

                if is_comment and ret == 0: # Open
                    target = item.getProperty('target')
                    info = self.odysee.resolve(target)[target]
                    signing_channel = info['signing_channel']
                    claim_id = target.split('#')[1]
                    CommentWindow(signing_channel['name'], signing_channel['claim_id'], claim_id, False, lbry=self.odysee, user_channel=get_user_channel(), selected_comment_id=item.getProperty('comment_hash'))

                start_offset = 1 if is_comment else 0

                if ret == start_offset + 0: # Mark as read/unread
                    id = item.getProperty('id')
                    self.odysee.notification_edit(id, 'is_read', not is_read)
                    if is_read:
                        self._add_unread_prefix(item)
                    else:
                        self._remove_unread_prefix(item)
                elif ret == start_offset + 1: # Delete
                    id = item.getProperty('id')
                    self.odysee.notification_delete(id)
                    pos = ccl.getSelectedPosition()
                    ccl.removeItem(pos)
                    if pos == ccl.size():
                        pos = pos - 1
                    ccl.selectItem(pos)
                elif ret == start_offset + 2: # Mark all as unread
                    ids = ''
                    for idx in range(0,ccl.size()):
                        li = ccl.getListItem(idx)
                        if li.getProperty('is_read') == 'False':
                            if ids != '':
                                ids += ','
                            ids += li.getProperty('id')

                    if ids != '':
                        self.odysee.notification_edit(ids, 'is_read', True)

                        for idx in range(0,ccl.size()):
                            li = ccl.getListItem(idx)
                            if li.getProperty('is_read') == 'False':
                                self._remove_unread_prefix(li)
                elif ret == start_offset + 3: # Mark all as read
                    ids = ''
                    for idx in range(0,ccl.size()):
                        li = ccl.getListItem(idx)
                        if li.getProperty('is_read') == 'True':
                            if ids != '':
                                ids += ','
                            ids += li.getProperty('id')

                    if ids != '':
                        self.odysee.notification_edit(ids, 'is_read', False)

                        for idx in range(0,ccl.size()):
                            li = ccl.getListItem(idx)
                            if li.getProperty('is_read') == 'True':
                                self._add_unread_prefix(li)
                elif ret == start_offset + 4: # Delete all
                    if Dialog().yesno('Delete all notifications', 'Are you sure?'):
                        ids = ''
                        for idx in range(0,ccl.size()):
                            li = ccl.getListItem(idx)
                            if ids != '':
                                ids += ','
                            ids += li.getProperty('id')
                        self.odysee.notification_delete(ids)
                        ccl.reset()
                elif ret == start_offset + 4: # Refresh
                    self.refresh()
        else:
            WindowXMLDialog.onAction(self, action)

    def refresh(self):
        self.last_selected_position = -1
        progressDialog = xbmcgui.DialogProgress()
        progressDialog.create(tr(30323), tr(30220))

        try:
            ccl = self.get_notification_control_list()
            ccl.reset()

            notifications = self.odysee.notification_list()
            progressDialog.update(100)
            progressDialog.close()

            for notification in notifications['data']:
                if notification['is_deleted']:
                    continue

                is_read = notification['is_read']

                target = None
                is_comment = notification['type'] == 'comment_replies'
                device = notification['notification_parameters']['device']
                if is_comment:
                    dynamic = notification['notification_parameters']['dynamic']
                    target = device['target']
                    title = f'{device["title"].split(" ")[0]} replied on "{dynamic["claim_title"]}"'
                    text = dynamic['comment']
                else:
                    title = device['title']
                    text = device['text']

                label = f'{title}: {text}'

                if not is_read:
                    label = f'{self._unread_label_prefix()}{label}'

                li = ListItem(label=label)
                li.setProperty('is_read', str(is_read))
                li.setProperty('is_comment', str(is_comment))
                li.setProperty('id', str(notification['id']))
                if is_comment:
                    li.setProperty('comment_hash', str(notification['notification_parameters']['dynamic']['hash']))
                if target:
                    li.setProperty('target', target)
                ccl.addItem(li)

                self.has_notifications = True

            if ccl.size() == 0:
                li = ListItem(label=tr(30322))
                ccl.addItem(li)
                self.has_notifications = False

        except Exception as e:
            progressDialog.update(100)
            progressDialog.close()
            raise e

        progressDialog.update(100)
        progressDialog.close()

    def get_notification_control_list(self) -> xbmcgui.ControlList:
        return typing.cast(xbmcgui.ControlList, self.getControl(1))

class NotificationWindow:
    def __init__(self, odysee: Odysee):
        window = NotificationWindowXML('addon-lbry-comments.xml', xbmcaddon.Addon().getAddonInfo('path'), 'Default', odysee=odysee)
        window.doModal();
        del window
