import typing
import xbmc
import xbmcaddon
import xbmcgui
from xbmcgui import Dialog, ListItem, WindowXMLDialog
from resources.lib.exception import PluginException, UserChannelDoesNotExist
from resources.lib.lbry import *
from resources.lib.local import clear_user_channel

tr = xbmcaddon.Addon().getLocalizedString

class CommentWindowXML(WindowXMLDialog):
    def __init__(self, *args, **kwargs):
        WindowXMLDialog.__init__(self, *args, **kwargs)
        self.channel_name = kwargs['channel_name']
        self.channel_id = kwargs['channel_id']
        self.claim_id = kwargs['claim_id']
        self.is_livestream = kwargs['is_livestream']
        self.lbry = kwargs['lbry']
        self.user_channel = kwargs['user_channel']
        if self.user_channel[0] == '' or self.user_channel[1] == '':
            self.user_channel = None
        self.last_selected_position = -1
        self.selected_comment_id = kwargs['selected_comment_id']

    def onInit(self):
        try:
            self.refresh()
        except UserChannelDoesNotExist as e:
            clear_user_channel()
            self.refresh()
            raise PluginException(f"User channel '{self.user_channel}' does not exist: Select a new user channel in the settings.")
        except PluginException as e:
            raise e

    def onAction(self, action):
        try:
            self._onAction(action)
        except UserChannelDoesNotExist as e:
            clear_user_channel()
            errmsg = "User channel '{self.user_channel}' does not exist: Select a new user channel in the settings."
            Dialog().ok(tr(30100), errmsg)
            xbmc.log("UserChannelDoesNotExist: " + str(e))
        except PluginException as e:
            Dialog().ok(tr(30100), str(e))
            xbmc.log("PluginException: " + str(e))
        except Exception as e:
            xbmc.log("Exception: " + str(e))

    def _onAction(self, action):
        if action == xbmcgui.ACTION_CONTEXT_MENU:
            # No user channel. Allow user to select an account or refresh.
            if not self.user_channel:
                ret = Dialog().contextmenu([tr(30240)])
                if ret == 0:
                    self.refresh()
                return

            # User channel selected. Allow comment manipulation.
            if self.user_channel:
                ccl = self.get_comment_control_list()
                selected_pos = ccl.getSelectedPosition()
                item = ccl.getSelectedItem()

                menu = []
                offsets = []
                offset = 0
                invalid_offset = 10000

                comment_id = item.getProperty('id')

                if item and not self.is_livestream:
                    if item.getProperty('my_vote') != str(0):
                        menu.append(tr(30228)) # Clear Vote
                        offsets.append(offset)
                        offset = offset + 1
                    else:
                        offsets.append(invalid_offset)

                    if item.getProperty('my_vote') != str(1):
                        menu.append(tr(30226)) # Like
                        offsets.append(offset)
                        offset = offset + 1
                    else:
                        offsets.append(invalid_offset)

                    if item.getProperty('my_vote') != str(-1):
                        menu.append(tr(30227)) # Dislike
                        offsets.append(offset)
                        offset = offset + 1
                    else:
                        offsets.append(invalid_offset)
                else:
                    offsets.append(invalid_offset)
                    offsets.append(invalid_offset)
                    offsets.append(invalid_offset)
                    offset = 0

                menu.append(tr(30221)) # New comment
                offsets.append(offset)
                offset = offset + 1

                if item:
                    if not self.is_livestream:
                        menu.append(tr(30222)) # Reply
                        offsets.append(offset)
                        offset = offset + 1
                    else:
                        offsets.append(invalid_offset)

                    if item.getProperty('channel_id') == self.user_channel[1]:

                        menu.append(tr(30223)) # Edit
                        offsets.append(offset)
                        offset = offset + 1

                        menu.append(tr(30224)) # Remove
                        offsets.append(offset)
                        offset = offset + 1
                    else:
                        offsets.append(invalid_offset)
                        offsets.append(invalid_offset)
                else:
                    offsets.append(invalid_offset)
                    offsets.append(invalid_offset)
                    offsets.append(invalid_offset)

                menu.append(tr(30240)) # Refresh
                offsets.append(offset)

                ret = Dialog().contextmenu(menu)

                if ret == offsets[0]: # Clear Vote
                    self.neutral(comment_id, item.getProperty('my_vote'))
                    item.setProperty('my_vote', str(0))
                    self.refresh_label(item)

                elif ret == offsets[1]: # Like
                    self.like(comment_id)
                    item.setProperty('my_vote', str(1))
                    self.refresh_label(item)

                elif ret == offsets[2]: # Dislike
                    self.dislike(comment_id)
                    item.setProperty('my_vote', str(-1))
                    self.refresh_label(item)

                elif ret == offsets[3]: # New Comment
                    comment = Dialog().input(tr(30221), type=xbmcgui.INPUT_ALPHANUM)
                    if comment:
                        comment_id = self.create_comment(comment)

                        # Remove 'No Comments' item
                        if ccl.size() == 1 and ccl.getListItem(0).getLabel() == tr(30230):
                            ccl.reset()

                        # Add new comment item
                        item = self.create_list_item(comment_id, self.user_channel[0], self.user_channel[1], comment, 0, 0, 0, 1)

                        ccl.addItem(item)
                        ccl.selectItem(ccl.size()-1)

                elif ret == offsets[4]: # Reply
                    comment = Dialog().input(tr(30222), type=xbmcgui.INPUT_ALPHANUM)
                    if comment:
                        comment_id = self.create_comment(comment, comment_id)

                        # Insert new item by copying the list (no XMBC method to allow a fast insertion).
                        newItems = []
                        for i in range(selected_pos+1):
                            newItems.append(self.copy_list_item(ccl.getListItem(i)))
                        newItems.append(self.create_list_item(comment_id, self.user_channel[0], self.user_channel[1], comment, 0, 0, int(item.getProperty('indent'))+1, 1))
                        for i in range(selected_pos+1, ccl.size()):
                            newItems.append(self.copy_list_item(ccl.getListItem(i)))

                        ccl.reset()
                        ccl.addItems(newItems)
                        ccl.selectItem(selected_pos+1)

                elif ret == offsets[5]: # Edit
                    id = item.getProperty('id');
                    comment = item.getProperty('comment')
                    comment = Dialog().input(tr(30223), type=xbmcgui.INPUT_ALPHANUM, defaultt=comment)
                    if comment:
                        self.edit_comment(id, comment)
                        item.setProperty('comment', comment)
                        self.refresh_label(item)

                elif ret == offsets[6]: # Change User
                    indentRemoved = item.getProperty('indent')
                    self.remove_comment(comment_id)
                    ccl.removeItem(selected_pos)

                    while True:
                        if selected_pos == ccl.size():
                            break
                        indent = ccl.getListItem(selected_pos).getProperty('indent')
                        if indent <= indentRemoved:
                            break
                        ccl.removeItem(selected_pos)

                    if selected_pos > 0:
                        ccl.selectItem(selected_pos-1)

                    if ccl.size() == 0:
                        ccl.addItem(ListItem(label=tr(30230)))

                elif ret == offsets[7]: # Refresh
                    self.refresh()

        else:
            WindowXMLDialog.onAction(self, action)

        # If an action changes the selected item position refresh the label
        ccl = self.get_comment_control_list()
        if self.last_selected_position != ccl.getSelectedPosition():
            if self.last_selected_position >= 0 and self.last_selected_position < ccl.size():
                oldItem = ccl.getListItem(self.last_selected_position)
                if oldItem:
                    self.refresh_label(oldItem, False)
            newItem = ccl.getSelectedItem()
            if newItem:
                self.refresh_label(newItem, True)
            self.last_selected_position = ccl.getSelectedPosition()

    def fetch_comment_list(self, page):
        return self.lbry.comment_list(self.channel_name, self.channel_id, self.claim_id, page=page)

    def fetch_react_list(self, comment_ids) -> Union[dict[str,Any],None]:
        num_comments = len(comment_ids)
        batch_size = 49
        comment_offset = 0
        js_aggregate = None

        while comment_offset < num_comments:
            ids = comment_ids[comment_offset:comment_offset+min(batch_size,num_comments-comment_offset)]
            comment_ids_str = ''
            for comment_id in ids:
                comment_ids_str += comment_id + ','
            js = self.lbry.reaction_list(comment_ids_str)
            if js_aggregate:
                js_aggregate['others_reactions'].update(js['others_reactions'])
                if 'my_reactions' in js:
                    js_aggregate['my_reactions'].update(js['my_reactions'])
            else:
                js_aggregate = js

            comment_offset = comment_offset + batch_size

        return js_aggregate

    def refresh(self):
        self.last_selected_position = -1
        progressDialog = xbmcgui.DialogProgress()
        progressDialog.create(tr(30219), tr(30220) + ' 1')

        try:
            ccl = self.get_comment_control_list()

            page = 1
            result = self.fetch_comment_list(page)
            if result == None:
                progressDialog.update(100)
                progressDialog.close()
                self.close()
                Dialog().ok(tr(30324), tr(30325))
                return

            total_pages = result['total_pages']

            if self.is_livestream:
                total_pages = min(total_pages, 5)

            while page < total_pages:
                if progressDialog.iscanceled():
                    break
                progressDialog.update(int(100.0*page/total_pages), tr(30220) + " %s/%s" % (page + 1, total_pages))
                page = page+1
                result['items'] += self.fetch_comment_list(page)['items']

            if 'items' in result:
                ccl.reset()
                items = result['items']

                if self.is_livestream:
                    for i in range(len(items)):
                        item = items[len(items)-i-1]
                        channel_name = item['channel_name']
                        channel_id = item['channel_id']
                        comment = item['comment']
                        comment_id = item['comment_id']
                        support_amount = item['support_amount']
                        is_fiat = item['is_fiat']
                        comment = comment.replace("\n", " ") # Prevent comment truncation at the first newline.
                        ccl.addItem(self.create_list_item(comment_id, channel_name, channel_id, comment, 0, 0, 0, 0, support_amount, is_fiat))

                    if ccl.size():
                        ccl.selectItem(ccl.size()-1)
                else:
                    # Grab the likes and dislikes.
                    comment_ids = []
                    for item in items:
                        comment_ids.append(item['comment_id'])

                    result = self.fetch_react_list(comment_ids)
                    assert(result != None)

                    others_reactions = result['others_reactions']
                    my_reactions = None
                    if 'my_reactions' in result:
                        my_reactions = result['my_reactions']

                    # Items are returned newest to oldest which implies that child comments are always before their parents.
                    # Iterate from oldest to newest comments building up a pre-order traversal ordering of the comment tree. Order
                    # the tree roots by decreasing score (likes-dislikes).
                    sort_indices = []
                    i = len(items)-1
                    while i >= 0:
                        item = items[i]
                        comment_id = item['comment_id']
                        if 'parent_id' in item and item['parent_id'] != 0:
                            for j in range(len(sort_indices)): # search for the parent in the sorted index list
                                sorted_item = items[sort_indices[j][0]]
                                indent = sort_indices[j][1]
                                if sorted_item['comment_id'] == item['parent_id']: # found the parent
                                    # Insert at the end of the subtree of the parent. Use the indentation to figure
                                    # out where the end is.
                                    while j+1 < len(sort_indices):
                                        if sort_indices[j+1][1] > indent: # Item with index j+1 is in the child subtree
                                            j = j+1
                                        else: # Item with index j+1 is not in the child subtree. Break and insert before this item.
                                            break
                                    sort_indices.insert(j+1, (i, indent+1, 0))
                                    break
                        else:
                            reaction = others_reactions[comment_id]
                            likes = reaction['like']
                            dislikes = reaction['dislike']
                            score = likes-dislikes
                            if my_reactions:
                                my_reaction = my_reactions[comment_id]
                                score += my_reaction['like'] - my_reaction['dislike']

                            j = 0
                            insert_index = len(sort_indices)
                            while j < len(sort_indices):
                                if sort_indices[j][1] == 0 and score > sort_indices[j][2]:
                                    insert_index = j
                                    break
                                j = j+1

                            sort_indices.insert(insert_index, (i, 0, score))

                        i -= 1

                    for (index,indent,score) in sort_indices:
                        item = items[index]
                        channel_name = item['channel_name']
                        channel_id = item['channel_id']
                        comment = item['comment']
                        comment_id = item['comment_id']
                        support_amount = item['support_amount']
                        is_fiat = item['is_fiat']
                        reaction = result['others_reactions'][comment_id]
                        likes = reaction['like']
                        dislikes = reaction['dislike']

                        if 'my_reactions' in result:
                            my_reaction = result['my_reactions'][comment_id]
                            my_vote = my_reaction['like'] - my_reaction['dislike']
                        else:
                            my_vote = 0

                        comment = comment.replace("\n", " ") # Prevent comment truncation at the first newline.

                        ccl.addItem(self.create_list_item(comment_id, channel_name, channel_id, comment, likes, dislikes, indent, my_vote, support_amount, is_fiat))
                        if self.selected_comment_id and comment_id == self.selected_comment_id:
                            self.last_selected_position = ccl.size() - 1

                    ccl.selectItem(self.last_selected_position)
            else:
                if ccl.size() == 0:
                    ccl.addItem(ListItem(label=tr(30230))) # No Comments
        except Exception as e:
            progressDialog.update(100)
            progressDialog.close()
            raise e

        progressDialog.update(100)
        progressDialog.close()

    def get_comment_control_list(self) -> xbmcgui.ControlList:
        return typing.cast(xbmcgui.ControlList, self.getControl(1))

    def create_list_item(self, comment_id, channel_name, channel_id, comment, likes, dislikes, indent, my_vote, support_amount=0, is_fiat=False):
        li = ListItem(label=self.create_label(channel_name, channel_id, likes, dislikes, comment, indent, my_vote, False, support_amount, is_fiat))
        li.setProperty('id', comment_id)
        li.setProperty('channel_name', channel_name)
        li.setProperty('channel_id', channel_id)
        li.setProperty('likes', str(likes))
        li.setProperty('dislikes', str(dislikes))
        li.setProperty('comment', comment)
        li.setProperty('indent', str(indent))
        li.setProperty('my_vote', str(my_vote))
        li.setProperty('support_amount', str(support_amount))
        li.setProperty('is_fiat', str(is_fiat))
        return li

    def copy_list_item(self, li):
        li_copy = ListItem(label=li.getLabel())
        li_copy.setProperty('id', li.getProperty('id'))
        li_copy.setProperty('channel_name', li.getProperty('channel_name'))
        li_copy.setProperty('channel_id', li.getProperty('channel_id'))
        li_copy.setProperty('likes', li.getProperty('likes'))
        li_copy.setProperty('dislikes', li.getProperty('dislikes'))
        li_copy.setProperty('comment', li.getProperty('comment'))
        li_copy.setProperty('indent', li.getProperty('indent'))
        li_copy.setProperty('my_vote', li.getProperty('my_vote'))
        li_copy.setProperty('support_amount', li.getProperty('support_amount'))
        li_copy.setProperty('is_fiat', li.getProperty('is_fiat'))
        return li_copy

    def refresh_label(self, li, selected=True):
        li.getProperty('id');
        channel_name = li.getProperty('channel_name')
        channel_id = li.getProperty('channel_id')
        likes = int(li.getProperty('likes'))
        dislikes = int(li.getProperty('dislikes'))
        comment = li.getProperty('comment')
        indent = int(li.getProperty('indent'))
        my_vote = int(li.getProperty('my_vote'))
        support_amount = float(li.getProperty('support_amount'))
        is_fiat = li.getProperty('is_fiat') == 'True'
        li.setLabel(self.create_label(channel_name, channel_id, likes, dislikes, comment, indent, my_vote, selected, support_amount, is_fiat))

    def create_label(self, channel_name, channel_id, likes, dislikes, comment, indent, my_vote, selected, support_amount=0.0, is_fiat=False):
        if self.user_channel and self.user_channel[1] == channel_id:
            color = 'red' if selected else 'green'
            channel_name = '[COLOR ' + color + ']' + channel_name + '[/COLOR]'

        if my_vote == 1:
            likes = '[COLOR green]' + str(likes+1) + '[/COLOR]'
            dislikes = str(dislikes)
        elif my_vote == -1:
            likes = str(likes)
            dislikes = '[COLOR green]' + str(dislikes+1) + '[/COLOR]'
        else:
            likes = str(likes)
            dislikes = str(dislikes)

        lilabel = channel_name
        if not self.is_livestream:
            lilabel += ' [COLOR orange]' + likes + '/' + dislikes + '[/COLOR]'

        if support_amount > 0.0:
            support_sym = '$' if is_fiat else 'L'
            lilabel += ' [COLOR yellow]' + '{}{:0.2f}'.format(support_sym, support_amount) + '[/COLOR]'

        lilabel += ' [COLOR white]' + comment + '[/COLOR]'

        padding = ''
        i = 0
        while i < indent:
            padding += '   '
            i = i+1
        lilabel = padding + lilabel

        return lilabel

    def create_comment(self, comment, parent_id=None):
        progressDialog = xbmcgui.DialogProgress()
        progressDialog.create(tr(30241), tr(30242))
        try:
            res = self.lbry.comment_create(self.claim_id, comment, parent_id)
            if not self.is_livestream:
                self.like(res['comment_id'])
        except Exception as e:
            progressDialog.close()
            raise e
        progressDialog.close()
        return res['comment_id']

    def edit_comment(self, comment_id, comment):
        return self.lbry.comment_edit(comment_id, comment)

    def remove_comment(self, comment_id):
        self.lbry.comment_abandon(comment_id)

    def react(self, comment_id, current_vote=0, type=None):
        # No vote to clear
        if current_vote == '0' and type == None:
            return

        remove = False
        if type != 'like' and type != 'dislike':
            remove = True
            type = 'dislike' if current_vote == '-1' else 'like'

        self.lbry.reaction_react(comment_id, type, remove)

    def like(self, comment_id):
        self.react(comment_id, type='like')

    def dislike(self, comment_id):
        self.react(comment_id, type='dislike')

    def neutral(self, comment_id, current_vote):
        self.react(comment_id, current_vote=current_vote)

class CommentWindow:
    def __init__(self, channel_name, channel_id, claim_id, is_livestream, lbry, user_channel, selected_comment_id=''):
        window = CommentWindowXML('addon-lbry-comments.xml', xbmcaddon.Addon().getAddonInfo('path'), 'Default', channel_name=channel_name, channel_id=channel_id, claim_id=claim_id, is_livestream=is_livestream, lbry=lbry, user_channel=user_channel, selected_comment_id=selected_comment_id)
        window.doModal();
        del window

