import os
import platform
import requests
import socket
import stat
import subprocess
import time
import urllib.parse
import urllib.request
from urllib.parse import unquote
import xbmc
import xbmcaddon
import xbmcgui
import xbmcvfs
import zipfile

from resources.lib.local import *
from resources.lib.lbry import *
from resources.lib.odysee import *

dl_links = {
    'Linux-x86_64' : ('https://github.com/lbryio/lbry-sdk/releases/download/v0.106.0/lbrynet-linux.zip', True),
    'Linux-armv7l' : ('https://slavegrid.com/kodi/lbrynet/v0.106.0/lbrynet-linux-armv7l.zip', False),
    'Linux-aarch64' : ('https://slavegrid.com/kodi/lbrynet/v0.106.0/lbrynet-linux-armv7l.zip', False),
    'Windows' : ('https://github.com/lbryio/lbry-sdk/releases/download/v0.106.0/lbrynet-windows.zip', True),
    'Darwin' : ('https://github.com/lbryio/lbry-sdk/releases/download/v0.106.0/lbrynet-mac.zip', True)
}

clean_cache_interval = 300 # Check every 5 minutes
predownload_interval = 300 # Check every 5 minutes
livestream_notification_interval = 30 # Check every minute
user_notification_interval = 30 # Check every minute
livestream_notification_display_ms = 25000
user_notification_display_ms = 25000
local_server_url = 'http://localhost:5279'

addon = xbmcaddon.Addon()
addon_path = addon.getAddonInfo('path')
tr = addon.getLocalizedString
lbrynet_process = None
profile_path = xbmcvfs.translatePath(addon.getAddonInfo('profile'))
bin_path = os.path.join(profile_path, 'bin')
daemon_config_path = os.path.join(profile_path, 'daemon_settings.yml')
lbrynet_path = os.path.join(bin_path, 'lbrynet')

def is_windows():
    return platform.system() == 'Windows'

def is_linux():
    return platform.system() == 'Linux'

def is_mac():
    return platform.system() == 'Darwin'

if is_windows():
    lbrynet_path += '.exe'

def get_ip():
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.settimeout(0)
    try:
        s.connect(('10.255.255.255', 1))
        ip = s.getsockname()[0]
    except Exception:
        ip = '127.0.0.1'
    finally:
        s.close()
    return ip

def get_default_config():
    return \
        f"data_dir: {profile_path}lbrynet\n" \
        f"wallet_dir: {profile_path}lbryum\n" \
        f"download_dir: {profile_path}download\n"

def refresh_configuration():
    config = ""
    with open(daemon_config_path, 'r') as f:
        for line in f:
            line.lstrip()
            if 'network_interface:' not in line and \
               'api:' not in line and \
               'streaming_server' not in line and \
               'use_upnp' not in line and \
               'share_usage_data' not in line: \
                config += line

    addon = xbmcaddon.Addon()

    if not addon.getSettingBool('allow_seeding'):
        config += 'network_interface: 127.0.0.1\n'

    if not addon.getSettingBool('enable_upnp'):
        config += 'use_upnp: false\n'

    if not addon.getSettingBool('share_usage_data'):
        config += 'share_usage_data: false\n'

    if xbmcaddon.Addon().getSettingBool('listen_to_external_connections'):
        # wait for the network interface to come up so we can grab the internal ip
        tries = 6
        ip = get_ip()
        while tries > 0:
            if ip == '127.0.0.1':
                xbmc.sleep(5)
                tries -= 1
                ip = get_ip()
            else:
                break

        config += f'api: 0.0.0.0:5279\n'
        config += f'streaming_server: {ip}:5280\n' # 0.0.0.0 causes the server to return links to localhost

    config += 'lbryum_servers:\n - a-hub1.odysee.com:50001\n'

    with open(daemon_config_path, 'w') as f:
        f.write(config)

    xbmc.log('Refreshed configuration')

def start_lbrynet():
    global lbrynet_process
    installed = False
    if os.path.exists(lbrynet_path):
        # kill any instances hanging around due to improper shutdown (Kodi crash)
        if is_windows():
            os.system('taskkill /im lbrynet.exe /f')
        elif is_linux() or is_mac():
            os.system('killall lbrynet')

        if not lbrynet_process:
            refresh_configuration()
            lbrynet_process = subprocess.Popen([lbrynet_path, 'start', f'--config={daemon_config_path}'])
            xbmc.log('Started lbrynet')
        installed = True
    else:
        installed = False

    xbmcaddon.Addon().setSettingBool('is_daemon_installed', installed)

def stop_lbrynet():
    global lbrynet_process
    if lbrynet_process:
        if is_windows():
            os.system('taskkill /im lbrynet.exe /f')
            xbmc.sleep(5000)
            lbrynet_process = None
            xbmc.log('Stopped lbrynet')
            return True
        else:
            lbrynet_process.terminate()
            try:
                lbrynet_process.wait(timeout=60)
                lbrynet_process = None
                xbmc.log('Stopped lbrynet')
                return True
            except subprocess.TimeoutExpired:
                return False

    return True

def restart_lbrynet():
    stop_lbrynet()
    start_lbrynet()
    return xbmcgui.Dialog().ok('LBRY', tr(30253)) # Restarted lbrynet

def should_replace_config():
    return xbmcgui.Dialog().yesno('LBRY', tr(30254)) # "Replace the existing daemon configuration with the default configuration?"

last_chunk = 0
t_last = 0.0
bitrate = 0.0

def download_url(url, progress_dialog):
    global last_chunk
    global t_last
    global bitrate
    last_chunk = 0
    t_last = time.time()

    def dl_report_hook(chunk, block_size, total_size):
        global last_chunk
        global t_last
        global bitrate

        if progress_dialog.iscanceled():
            raise Exception()

        t_cur = time.time()
        t_delta = t_cur - t_last
        if t_delta > 1.0:
            sz_delta = (chunk-last_chunk)*block_size
            bitrate = sz_delta/t_delta
            t_last = t_cur
            last_chunk = chunk

        o = urllib.parse.urlparse(url)
        progress_dialog.update(100*chunk*block_size//total_size, f'{tr(30255)} {o.hostname}\n\n{chunk*block_size/(1000*1000):.2f}/{total_size/(1000*1000):.2f} MB @ {bitrate/(1000*1000):.2f} MB/s')

    return urllib.request.urlretrieve(url, reporthook=dl_report_hook)

def install_lbrynet():
    xbmc.log('Installing lbrynet binary')
    system = platform.system()
    if is_linux():
        machine = platform.machine()
        (dl_link,is_official) = dl_links[f"{system}-{machine}"]
    else:
        (dl_link,is_official) = dl_links[system]

    if not is_official:
        if not xbmcgui.Dialog().yesno('LBRY', tr(30256)):
            return

    progress_dialog = xbmcgui.DialogProgress()
    progress_dialog.create('LBRY', tr(30257))
    xbmc.sleep(1000)

    try:
        dl_filename = download_url(dl_link, progress_dialog)[0]
    except:
        progress_dialog.update(100, tr(30262))
    else:
        try:
            with zipfile.ZipFile(dl_filename, 'r') as zip_ref:
                if stop_lbrynet():
                    if not os.path.exists(bin_path):
                        os.mkdir(bin_path)
                    zip_ref.extractall(bin_path)
                    st = os.stat(lbrynet_path)
                    os.chmod(lbrynet_path, st.st_mode | stat.S_IEXEC)

                    if not os.path.exists(daemon_config_path) or should_replace_config():
                        with open(daemon_config_path, 'w+') as f:
                            f.write(get_default_config())

                    progress_dialog.update(100, tr(30258))
                    start_lbrynet()
                    xbmcaddon.Addon().setSetting('lbry_api_url', local_server_url)
                    xbmcaddon.Addon().setSettingBool('use_odysee', False)

                    # Give 10 seconds to spin up so the users aren't given an exception about unloaded components when they try to play.
                    xbmc.sleep(10000)

                    # Skip this message if the user hits cancel as its only there for feedback.
                    if not progress_dialog.iscanceled():
                        progress_dialog.update(100, tr(30259))
                        xbmc.sleep(4000)

                    xbmc.log('Installed lbrynet binary')
                else:
                    progress_dialog.update(100, tr(30260))
                    xbmc.log('Failed to install lbrynet binary')
        except Exception as e:
            progress_dialog.update(100, f'{tr(30261)}: "{str(e)}"')
            xbmc.sleep(5000)
            xbmc.log('Failed to install lbrynet binary')

    progress_dialog.close()
    urllib.request.urlcleanup()

def uninstall_lbrynet():
    progress_dialog = xbmcgui.DialogProgress()
    progress_dialog.create('LBRY', tr(30263))
    xbmc.sleep(500)
    stop_lbrynet()
    if os.path.exists(lbrynet_path):
        os.remove(lbrynet_path)
    addon = xbmcaddon.Addon()
    addon.setSettingBool('is_daemon_installed', False)
    addon.setSetting('lbry_api_url', '')
    progress_dialog.update(100, tr(30264))
    xbmc.sleep(2000)
    progress_dialog.close()

def process_trigger_file(filename, hook):
    trigger_file_path = os.path.join(addon_path, filename)
    if os.path.exists(trigger_file_path):
        if hook:
            hook()
        os.remove(trigger_file_path)

def get_size(path):
    total_size = 0
    for res in os.walk(path):
        filenames = res[2]
        for f in filenames:
            dirpath = res[0]
            fp = os.path.join(dirpath, f)
            if not os.path.islink(fp):
                total_size += os.path.getsize(fp)
    return total_size

def blob_cache_size():
    blob_path = os.path.join(xbmcvfs.translatePath(xbmcaddon.Addon().getAddonInfo('profile')), 'lbrynet/blobfiles')
    return get_size(blob_path)

def fetch_file_list():
    page = 1
    items = []
    while 1:
        json = { 'method' : 'file_list', 'params' : {'page' : page } }
        result = requests.post(local_server_url, json=json)
        result.raise_for_status()
        rjson = result.json()
        if 'result' in rjson and 'items' in rjson['result']:
            items += rjson['result']['items']
            if 'total_pages' in rjson['result'] and page < rjson['result']['total_pages']:
                page = page + 1
            else:
                break
        else:
            break
    return items

def clean_cache():
    xbmc.log('Cleaning cache')
    try:
        file_items = fetch_file_list()

        xbmc.log(f'Cache size: {blob_cache_size()/(1024**2):.2f}MiB')
        delete_item_idx = 0
        while blob_cache_size() > xbmcaddon.Addon().getSettingInt('max_cache_size')*(1024**3):
            if delete_item_idx >= len(file_items):
                break
            item_to_delete = file_items[delete_item_idx]
            json = { 'method' : 'file_delete', 'params' : { 'claim_id' : item_to_delete['claim_id'], 'delete_from_download_dir' : True } }
            requests.post(local_server_url, json=json)
            xbmc.log(f"Deleted blobs for file {item_to_delete['metadata']['title']} (Cache size: {blob_cache_size()/(1024**2):.2f}MiB)")
            delete_item_idx += 1
    except Exception as e:
        xbmc.log(f'Failed to clean cache: {str(e)}')
    xbmc.log('Finished cleaning cache')

def predownload():
    xbmc.log('Predownloading recent videos')
    try:
        temp_download_path = os.path.join(profile_path, 'lbrynet/temp_download')

        file_items = fetch_file_list()

        if os.path.exists(temp_download_path):
            for item in file_items:
                if item['blobs_remaining'] == 0 and item['download_path'] != None:
                    if temp_download_path in item['download_path']:
                        os.remove(item['download_path'].encode('utf-8'))
                        xbmc.log(f"Deleted file {item['download_path']}")
        else:
            os.mkdir(temp_download_path)

        lbry = Lbry('http://localhost:5279')
        channels = lbry.subscriptions()
        if len(channels) != 0:
            channel_ids = []
            for channel in channels:
                claim_id = channel[1]
                channel_ids.append(claim_id)
            preload_video_count = xbmcaddon.Addon().getSettingInt('preload_recent_video_count')
            json = {
                'method': 'claim_search',
                'params': { 'page': 1, 'page_size': preload_video_count, 'order_by': 'release_time', 'channel_ids': channel_ids }
                }
            result = requests.post(local_server_url, json=json)
            result.raise_for_status()
            rjson = result.json()
            if 'result' in rjson:
                uri_list = []
                for item in rjson['result']['items']:
                    # only consider video streams
                    if item['value_type'] != 'stream' or 'stream_type' not in item['value'] or item['value']['stream_type'] != 'video':
                        continue

                    # skip files already added
                    append = True
                    for file_item in file_items:
                        if file_item['claim_id'] == item['claim_id'] and (file_item['blobs_remaining'] == 0 or file_item['stopped'] != False):
                            append = False
                            break
                    if append:
                        uri_list.append(f"{item['name']}#{item['claim_id']}")

                uri_list = reversed(uri_list)
                for uri in uri_list:
                    json = {
                        'method': 'get',
                        'params': {'uri': uri, 'save_file': True, 'download_directory': temp_download_path }
                        }
                    requests.post(local_server_url, json=json)
                    xbmc.log(f'Predownloading {uri}')
    except Exception as e:
        xbmc.log(f'Failed to predownload: {str(e)}')
    xbmc.log('Finished predownloading recent videos')

displayed_livestream_notifications = []

def display_livestream_notifications():
    global displayed_livestream_notifications
    xbmc.log("Displaying livestream notifications")
    try:
        if ADDON.getSettingBool('use_odysee'):
            odysee = Odysee()
            try:
                with open_file('odysee_login', 'r') as f:
                    odysee.loadf(f)
            except:
                pass

            if odysee.signed_in:
                channels = odysee.subscriptions()
            else:
                channels = load_local_channel_subs()

            lbry = odysee
        else:
            lbry_api_url = unquote(ADDON.getSetting('lbry_api_url'))
            lbry = Lbry(lbry_api_url)
            channels = lbry.subscriptions()

        if len(channels) != 0:
            channel_ids = []
            for channel in channels:
                claim_id = channel[1]
                channel_ids.append(claim_id)

            res = lbry.claim_search(channel_ids = channel_ids, order_by = Lbry.ORDER_BY_RELEASE_TIME, live=True)

            livestream_json = get_all_livestreams_json()
            for item in res['items']:
                if is_livestreaming(livestream_json, item['claim_id']):
                    if item['claim_id'] not in displayed_livestream_notifications:
                        channel_name = item['signing_channel']['name']
                        if 'value' in item['signing_channel'] and 'title' in item['signing_channel']['value']:
                            channel_title = item['signing_channel']['value']['title']
                            channel_name = channel_title if channel_title.strip() != '' else channel_name
                        title = item['value']['title']

                        thumbnail_uri = xbmcgui.NOTIFICATION_INFO
                        if 'value' in item['signing_channel'] and 'thumbnail' in item['signing_channel']['value'] and 'url' in item['signing_channel']['value']['thumbnail']:
                            thumbnail_uri = optimized_thumbnail_uri(item['signing_channel']['value']['thumbnail']['url'], size=(256,0))

                        xbmcgui.Dialog().notification(tr(30279), f'[COLOR orange]{channel_name} | {title}[/COLOR]', thumbnail_uri, livestream_notification_display_ms, True)
                        xbmc.log(f'Displayed notification {channel_name} | {title}')
                        displayed_livestream_notifications.append(item['claim_id'])

            # keep last 20 notifications
            displayed_livestream_notifications = displayed_livestream_notifications[-20:]
    except Exception as e:
        xbmc.log(f'Failed to display livestream notifications: {str(e)}')

    xbmc.log("Displayed livestream notifications")

displayed_user_notifications = []

def display_user_notifications():
    global displayed_user_notifications
    odysee = Odysee()
    try:
        with open_file('odysee_login', 'r') as f:
            odysee.loadf(f)
    except:
        return

    if not odysee.signed_in:
        return

    xbmc.log("Displaying Odysee user notifications")
    notifications = odysee.notification_list()
    for notification in notifications['data']:
        id = notification['id']
        if not notification['is_seen'] and id not in displayed_user_notifications:
            thumbnail_uri = xbmcgui.NOTIFICATION_INFO
            device = notification['notification_parameters']['device']
            if notification['type'] == 'comment_replies':
                dynamic = notification['notification_parameters']['dynamic']
                title = f'{device["title"].split(" ")[0]} replied on "{dynamic["claim_title"]}"'
                text = dynamic['comment']
                author_thumbnail = dynamic['comment_author_thumbnail']
                if author_thumbnail == '':
                    author_thumbnail = 'https://spee.ch/spaceman-png:2.png'
                thumbnail_uri = optimized_thumbnail_uri(author_thumbnail, size=(256,0))
            else:
                title = device['title']
                text = device['text']
            xbmcgui.Dialog().notification(title, f'[COLOR orange]{text}[/COLOR]', thumbnail_uri, user_notification_display_ms, True)
            odysee.notification_edit(id, 'is_seen', True)
            displayed_user_notifications.append(id)

    # keep last 20 notifications
    displayed_user_notifications = displayed_user_notifications[-20:]
    xbmc.log("Displayed Odysee user notifications")

class Monitor(xbmc.Monitor):
    def __init__(self):
        xbmc.Monitor.__init__(self)

        # Restart server if any of these settings are changed
        self.restart_keys = [ 'allow_seeding', 'listen_to_external_connections', 'enable_upnp', 'share_usage_data' ]
        self.last_restart_key_values = {}
        for key in self.restart_keys:
            self.last_restart_key_values[key] = xbmcaddon.Addon().getSettingBool(key)

    def onSettingsChanged(self):
        if not os.path.exists(lbrynet_path):
            return

        trigger_restart = False
        for key in self.restart_keys:
            val = xbmcaddon.Addon().getSettingBool(key)
            if self.last_restart_key_values[key] != val:
                trigger_restart = True
                self.last_restart_key_values[key] = val

        if trigger_restart:
            restart_lbrynet()

if __name__ == '__main__':
    start_lbrynet()

    time_since_last_cache_clean = time.time()
    time_since_last_predownload = time_since_last_cache_clean
    time_since_last_livestream_notification = 0
    time_since_last_user_notification = 0

    monitor = Monitor()
    while not monitor.abortRequested():
        if monitor.waitForAbort(2):
            process_trigger_file('install-lbrynet', None)
            process_trigger_file('uninstall-lbrynet', None)
            process_trigger_file('restart-lbrynet', None)
            stop_lbrynet()
            break

        if xbmcaddon.Addon().getSettingBool('display_livestream_notifications'):
            if time.time() - time_since_last_livestream_notification > livestream_notification_interval:
                display_livestream_notifications()
                time_since_last_livestream_notification = time.time()

        if ADDON.getSettingBool('use_odysee') and xbmcaddon.Addon().getSettingBool('display_user_notifications'):
            if time.time() - time_since_last_user_notification > user_notification_interval:
                display_user_notifications()
                time_since_last_user_notification = time.time()

        if lbrynet_process:
            if time.time() - time_since_last_cache_clean > clean_cache_interval:
                clean_cache()
                time_since_last_cache_clean = time.time()

            if xbmcaddon.Addon().getSettingBool('predownload_recent_videos'):
                if time.time() - time_since_last_predownload > predownload_interval:
                    predownload()
                    time_since_last_predownload = time.time()

        process_trigger_file('install-lbrynet', install_lbrynet)
        process_trigger_file('uninstall-lbrynet', uninstall_lbrynet)
        process_trigger_file('restart-lbrynet', restart_lbrynet)
