# -*- coding: utf-8 -*-
import requests
import xbmcaddon
import xbmcgui
import xbmcvfs

from typing import Any, Union

ADDON = xbmcaddon.Addon()
tr = ADDON.getLocalizedString

Channel = tuple[str,str]

def get_user_channel() -> Channel:
    if ADDON.getSettingBool('use_odysee'):
        return (ADDON.getSettingString('odysee_user_channel_name'),
        ADDON.getSettingString('odysee_user_channel_id'))
    else:
        return (ADDON.getSettingString('lbry_user_channel_name'),
        ADDON.getSettingString('lbry_user_channel_id'))

def set_user_channel(channel_name: str, channel_id: str):
    if ADDON.getSettingBool('use_odysee'):
        ADDON.setSettingString('odysee_user_channel_name', channel_name)
        ADDON.setSettingString('odysee_user_channel_id', channel_id)
    else:
        ADDON.setSettingString('lbry_user_channel_name', channel_name)
        ADDON.setSettingString('lbry_user_channel_id', channel_id)

def clear_user_channel():
    if ADDON.getSettingBool('use_odysee'):
        ADDON.setSettingString('odysee_user_channel_name', '')
        ADDON.setSettingString('odysee_user_channel_id', '')
    else:
        ADDON.setSettingString('lbry_user_channel_name', '')
        ADDON.setSettingString('lbry_user_channel_id', '')

def get_profile_path(rpath):
    return xbmcvfs.translatePath(ADDON.getAddonInfo('profile') + rpath)

def create_profile_directory_if_missing():
    profile_path = ADDON.getAddonInfo('profile')
    if not xbmcvfs.exists(profile_path):
        xbmcvfs.mkdir(profile_path)

def open_file(name, mode):
    if mode == 'w':
        create_profile_directory_if_missing()
    return xbmcvfs.File(get_profile_path(name), mode)

def delete_file(name):
    return xbmcvfs.delete(get_profile_path(name))

def file_exists(name):
    return xbmcvfs.exists(get_profile_path(name))

def load_local_channel_subs() -> list[tuple[str,str]]:
    channels = []
    try:
        with open_file('channel_subs', 'r') as f:
            lines = f.readBytes().decode('utf-8')
            for line in lines.split('\n'):
                items = line.split('#')
                if len(items) < 2:
                    continue
                channels.append((items[0],items[1]))
    except Exception as e:
        xbmcgui.Dialog().notification(tr(30104), str(e), xbmcgui.NOTIFICATION_ERROR)
    return channels

def save_local_channel_subs(channels):
    def sort_func(val):
        return val[0].lower()
    try:
        with open_file('channel_subs', 'w') as f:
            channels.sort(key=sort_func)
            for (name, claim_id) in channels:
                f.write(bytearray(name.encode('utf-8')))
                f.write('#')
                f.write(bytearray(claim_id.encode('utf-8')))
                f.write('\n')
    except Exception as e:
        xbmcgui.Dialog().notification(tr(30104), str(e), xbmcgui.NOTIFICATION_ERROR)

def get_all_livestreams_json():
    try:
        r = requests.get('https://api.odysee.live/livestream/all')
        r.raise_for_status()
        return r.json()
    except:
        return {}

def is_livestreaming(livestream_json: dict[str,Any], claim_id: str) -> bool:
    if 'data' in livestream_json:
        data = livestream_json['data']
        for item in data:
            if item['ActiveClaim']['ClaimID'] == claim_id:
                return item['Live']
    return False

def optimized_thumbnail_uri(uri: str, size: Union[tuple[int,int],None]=None, quality: Union[int,None]=None) -> str:
    s = 'https://thumbnails.odycdn.com/optimize/'
    if size:
        s += f's:{size[0]}:{size[1]}/'
    if quality:
        s += f'quality:{quality}/'
    return s + 'plain/' + uri
