#from local import call_rpc
import requests
import json
from typing import Union

from resources.lib.lbry import *

class Odysee(Lbry):
    PROXY_URL = 'https://api.na-backend.odysee.com/api/v1/proxy'
    ODYSEE_APP_ID = 'odyseecom692EAWhtoqDuAfQ6KHMXxFxt8tkhmt7sfprEMHWKjy5hf6PwZcHDV542V'

    def __init__(self, user_channel_name='', user_channel_id=''):
        super().__init__(Odysee.PROXY_URL, local=False, user_channel_name=user_channel_name, user_channel_id=user_channel_id)
        self.signed_in: bool = False
        self.email: Union[str,None] = None

    def new(self) -> None:
        """ Create a new user. """
        params = {
            'auth_token': '',
            'language': 'en',
            'app_id': Odysee.ODYSEE_APP_ID
        }

        retries = 5
        while 1:
            result = self.odysee_rpc('/user/new', params=params, raise_exception=False)
            if result.ok:
                rjson = result.json()
                if rjson and rjson['success']:
                    self.auth_token = rjson['data']['auth_token']
                    self.init_request_headers(self.auth_token)
                break
            else:
                # Try again after a short delay. Odysee rate limiting most likely.
                if result.status_code == 503 and retries > 0:
                    time.sleep(1)
                    retries = retries - 1
                    continue

                # Check if an error was provided by Odysee
                if result.json():
                    if 'error' in result.json():
                        raise PluginException(result.json()['error'])

                # Generic response code error
                result.raise_for_status();

    def load(self, path: str) -> None:
        """ Load from file """
        try:
            with open(path, 'r') as f:
                self.loadf(f)
        except FileNotFoundError:
            pass

    def loadf(self, f) -> None:
        """ Load from file handle """
        js = json.loads(f.read())
        if 'auth_token' in js:
            self.auth_token = js['auth_token']
        if 'email' in js:
            self.email = js['email']
            self.signed_in = True
        else:
            if self.email:
                self.email = None
            self.signed_in = False
        self.init_request_headers(self.auth_token)

    def save(self, path: str) -> None:
        """ Save to file """
        with open(path, 'w') as f:
            self.savef(f)

    def savef(self, f) -> None:
        """ Save from file handle """
        dict = {
                'auth_token' : self.auth_token,
                }
        if self.signed_in:
            dict['email'] = self.email
        f.write(json.dumps(dict))

    def user_exists(self, email: str) -> Any:
        """ Checks if the user exists based upon email """
        params = {
            'auth_token': self.auth_token,
            'email': email,
        }
        result = self.odysee_rpc('/user/exists', params, raise_exception=False)
        rjson = result.json()
        if not rjson:
            result.raise_for_status();
        return rjson['success']

    def signin(self, email: str, password: str) -> tuple[bool,str]:
        """ Attempt to sign into Odysee """
        if not self.user_exists(email):
            return (False, 'user does not exist')

        params = {
                'auth_token': self.auth_token,
                'email': email,
                'password': password,
                }
        retries = 5
        while 1:
            result = self.odysee_rpc('/user/signin', params, raise_exception=False)
            if result.ok:
                break
            else:
                # Try again after a short delay. Odysee rate limiting most likely.
                if result.status_code == 503 and retries > 0:
                    time.sleep(1)
                    retries = retries - 1
                    continue

                rjson = result.json()
                if rjson and 'error' in rjson:
                    return (False, rjson['error'])

                result.raise_for_status();

        rjson = result.json()
        self.signed_in = rjson and rjson['success']
        if self.signed_in:
            self.email = email
        return (self.signed_in, rjson['error'] if 'error' in rjson else '')

    def signout(self) -> None:
        """ Sign out """
        params = { 'auth_token': self.auth_token }
        self.odysee_rpc('/user/signout', params)
        self.signed_in = False
        self.email = ''

    def me(self) -> dict[str,Any]:
        """ Get information about the user """
        if self.signed_in:
            params = { 'auth_token': self.auth_token }
            return self.odysee_rpc('/user/me', params=params).json()
        else:
            raise PluginException("Cannot get information about user: Not signed in.")

    def subscribe(self, channel_name: str, claim_id: str) -> None:
        """ Subscribe to a channel """
        if self.signed_in:
            if channel_name:
                params = {
                        'auth_token': self.auth_token,
                        'channel_name': channel_name,
                        'claim_id': claim_id,
                        'notifications_disabled': 'true',
                        }
                rjson = self.odysee_rpc('/subscription/new', params).json()
                if rjson and rjson['success']:
                    super().subscribe(channel_name, claim_id)
        else:
            raise PluginException("Cannot subscribe: Not signed in.")

    def unsubscribe(self, channel_name: str, claim_id: str) -> None:
        """ Unsubscribe to a channel """
        if self.signed_in:
            if claim_id:
                params = {
                        'auth_token': self.auth_token,
                        'claim_id': claim_id,
                        }
                rjson = self.odysee_rpc('/subscription/delete', params).json()
                if rjson and rjson['success']:
                    super().unsubscribe(channel_name, claim_id)
        else:
            raise PluginException("Cannot unsubscribe: Not signed in.")

    def claim_reaction_react(self, claim_ids: str, type: str, remove: bool=False) -> LbryPayload:
        """ React to a claim """
        if self.signed_in:
            params: LbryPayload = {
                    'auth_token': self.auth_token,
                    'claim_ids': claim_ids,
                    'type': type
                    }
            if type == 'like':
                params['clear_types'] = 'dislike'
            elif type == 'dislike':
                params['clear_types'] = 'like'
            if remove:
                params['remove'] = 'true'

            return self.odysee_rpc('/reaction/react', params).json()
        else:
            raise PluginException("Cannot react to a claim: Not signed in.")

    def claim_reaction_list(self, claim_ids: str) -> LbryPayload:
        """ List claim reactions """
        params: LbryPayload = {
                'claim_ids': claim_ids,
                }

        if self.signed_in:
            params['auth_token'] = self.auth_token,

        return self.odysee_rpc('/reaction/list', params).json()

    def init_request_headers(self, auth_token) -> None:
        """ Init the LBRY API request headers when using the Odysee proxy """
        self.headers = {'content-type' : 'application/json', 'x-lbry-auth-token' : auth_token }
