import requests
import time
import uuid
from resources.lib.exception import *
from resources.lib.exception import PluginException
from typing import Any,Tuple,Union
from urllib.parse import quote_plus,unquote_plus

LbryPayload = dict[str,Any]
Headers = dict[str,str]
Channel = Tuple[str, str]

class Lbry:
    ORDER_BY_RELEASE_TIME = 'release_time'
    ORDER_BY_TRENDING_GLOBAL = 'trending_global'
    ORDER_BY_SUPPORT_AMOUNT = 'support_amount'
    ODYSEE_COMMENT_API_URL = 'https://comments.odysee.com/api/v2'
    ODYSEE_API_HOSTNAME = 'api.odysee.com'

    def __init__(self, lbry_api_url: str, local: bool=True, user_channel_name: str = '', user_channel_id: str = ''):
        self.lbry_api_url: str = lbry_api_url
        self.local: bool = local
        self.headers : Headers = {}
        self.set_user_channel(user_channel_name, user_channel_id)

    def set_user_channel(self, channel_name: str, channel_id: str) -> None:
        """ Set a user-owned channel """
        if channel_name != '' and channel_id != '':
            self.user_channel_name: str = channel_name
            self.user_channel_id: str = channel_id
        else:
            self.user_channel_name: str = ''
            self.user_channel_id: str = ''

    def is_user_channel_set(self) -> bool:
        """ Return true if the user has set an owned channel. Required for commenting. """
        return self.user_channel_name != '' and self.user_channel_id != ''

    def subscribe(self, channel_name: str, claim_id: str) -> None:
        """ Subscribe to a channel """
        prefs = self.preferences_get()

        try:
            following = prefs[self.preferences_key()]['value']['following']
        except KeyError:
            raise PluginException('Could not subscribe: "following" key missing from lbrynet preferences schema')
        try:
            subscriptions = prefs[self.preferences_key()]['value']['subscriptions']
        except KeyError:
            raise PluginException('Could not subscribe: "subscriptions" key missing from lbrynet preferences schema')

        uri = f'lbry://{channel_name}#{claim_id}'

        # insert into following
        found = False
        for follow in following:
            if uri == follow['uri']:
                found = True
                break
        if not found:
            following.append({
                'uri' : f'lbry://{channel_name}#{claim_id}'
                })

        # insert into subscriptions
        found = False
        for subscription in subscriptions:
            if uri == subscription:
                found = True

        if not found:
            subscriptions.append(uri)

        self.preferences_set(prefs[self.preferences_key()])

    def unsubscribe(self, channel_name: str, claim_id: str) -> None:
        """ Unsubscribe from a channel """
        prefs = self.preferences_get()
        try:
            following: list[dict] = prefs[self.preferences_key()]['value']['following']
        except KeyError:
            raise PluginException('Could not subscribe: "following" key missing from lbrynet preferences schema')
        try:
            subscriptions: list[str] = prefs[self.preferences_key()]['value']['subscriptions']
        except KeyError:
            raise PluginException('Could not subscribe: "subscriptions" key missing from lbrynet preferences schema')

        uri = f'lbry://{channel_name}#{claim_id}'
        for idx in range(len(following)):
            if following[idx]['uri'] == uri:
                del following[idx]
                break

        for idx in range(len(subscriptions)):
            if subscriptions[idx] == uri:
                del subscriptions[idx]
                break

        self.preferences_set(prefs[self.preferences_key()])

    def subscriptions(self) -> list[Channel]:
        """ Return subscriptions """
        prefs = self.preferences_get()
        subscriptions = []
        try:
            for entry in prefs[self.preferences_key()]['value']['subscriptions']:
                channel = entry.strip('lbry://').split('#')
                subscriptions.append((channel[0], channel[1]))
        except KeyError:
            raise PluginException(f'Could not get subscriptions: Unknown preferences schema.')

        subscriptions.sort(key=lambda x: x[0].lower())
        return subscriptions

    def owned_channel_list(self, page=1, page_size=50) -> LbryPayload:
        """ Return a list of owned channels """
        params: LbryPayload = {'page' : page, 'page_size': page_size}
        return self.lbrynet_rpc('channel_list', params)

    def resolve(self, urls : Union[str, list[str]]) -> LbryPayload:
        """ Resolve LBRY urls to their claim datas """
        return self.lbrynet_rpc('resolve', {'urls': urls})

    def get(self, uri: str, save_file: bool=False, lbry_streaming_url_override=None) -> LbryPayload:
        """ Return streaming url """
        return self.lbrynet_rpc('get', {'uri': uri, 'save_file': save_file}, lbry_api_url_override=lbry_streaming_url_override)

    def claim_search(self, text: Union[str,None]=None, channel: Union[str,None]=None, channel_ids: Union[list[str],None]=None, page: int=1, page_size: int=100, order_by: str=ORDER_BY_TRENDING_GLOBAL, claim_type: Union[list[str],None]=None, stream_types: Union[list[str],None]=None, live: bool=False) -> LbryPayload:
        """ Search for claims """
        params: LbryPayload = {'page': page, 'page_size': page_size, 'order_by': order_by }
        if text:
            params['text'] = text
        if channel:
            params['channel'] = channel
        if channel_ids:
            params['channel_ids'] = channel_ids
        if live:
            params['has_no_source'] = True
            params['limit_claims_per_channel'] = 2
            params['no_totals'] = True
            params['claim_type'] = ['stream']
        else:
            if stream_types:
                params['stream_types'] = stream_types
            if claim_type:
                params['claim_type'] = claim_type
        return self.lbrynet_rpc('claim_search', params)

    def account_list(self) -> LbryPayload:
        """ Return account list """
        return self.lbrynet_rpc('account_list')

    def comment_list(self, channel_name, channel_id, claim_id, page=1, page_size=50) -> LbryPayload:
        """ Return a list of comments for a claim """
        return self.odysee_comment_rpc('comment.List', params={"page":page,"page_size":page_size,'include_replies':True,'visible':False,'hidden':False,'top_level':False,'channel_name':channel_name,'channel_id':channel_id,'claim_id':claim_id,'sort_by':0})

    def comment_create(self, claim_id: str, comment: str, parent_id: Union[str,None]=None) -> LbryPayload:
        """ Create a comment """
        if self.is_user_channel_set():
            params: LbryPayload = { 'claim_id' : claim_id, 'comment' : comment, 'channel_id' : self.user_channel_id }
            if parent_id:
                params['parent_id'] = parent_id
            self.sign_params(comment, params)
            return self.odysee_comment_rpc('comment.Create', params)
        else:
            raise PluginException("Cannot create comment: No user set.")

    def comment_edit(self, comment_id: str, comment: str) -> LbryPayload:
        """ Edit a comment """
        if self.is_user_channel_set():
            params: LbryPayload = { 'comment_id' : comment_id, 'comment' : comment }
            self.sign_params(comment, params)
            return self.odysee_comment_rpc('comment.Edit', params)
        else:
            raise PluginException("Cannot edit comment: No user set.")

    def comment_abandon(self, comment_id: str) -> LbryPayload:
        """ Remove a comment """
        if self.is_user_channel_set():
            params: LbryPayload = { 'comment_id' : comment_id }
            self.sign_params(comment_id, params)
            return self.odysee_comment_rpc('comment.Abandon', params)
        else:
            raise PluginException("Cannot abandon comment: No user set.")

    def reaction_list(self, comment_ids: str) -> LbryPayload:
        """ Return comment reactions """
        params: LbryPayload = { 'comment_ids' : comment_ids }

        if self.is_user_channel_set():
            params['channel_name'] = self.user_channel_name
            params['channel_id'] = self.user_channel_id
            self.sign_params(self.user_channel_name, params)

        return self.odysee_comment_rpc('reaction.List', params=params)

    def reaction_react(self, comment_id: str, type: str, remove: bool=False) -> LbryPayload:
        """ React to a comment """
        if self.is_user_channel_set():
            params: LbryPayload = { 'comment_ids' : comment_id,
                    'channel_name' : self.user_channel_name,
                    'channel_id' : self.user_channel_id,
                    'type' : type,
                    }
            if type == 'like':
                params['clear_types'] = 'dislike'
            elif type == 'dislike':
                params['clear_types'] = 'like'
            if remove:
                params['remove'] = True
            self.sign_params(self.user_channel_name, params)
            return self.odysee_comment_rpc('reaction.React', params)
        else:
            raise PluginException("Cannot react to comment: No user set.")

    def union_collections(self, value):
        return value['builtinCollections'] | value['unpublishedCollections'] | value['editedCollections'] | value['updatedCollections']

    def collections_list(self) -> list[tuple[str,str]]:
        """ Return a list of collection (name, id) tuples """
        value = self.preferences_get()[self.preferences_key()]['value']
        collections: dict[str,Any] = self.union_collections(value)
        c = []
        for key in collections:
            c.append((collections[key]['name'], collections[key]['id']))
        return c

    def collection_items(self, collection_id) -> list[str]:
        """ Return the items of a collection """
        value = self.preferences_get()[self.preferences_key()]['value']
        collections: dict[str,Any] = self.union_collections(value)
        entries = []
        if collection_id in collections:
            for item in collections[collection_id]['items']:
                entries.append(item.strip('lbry://'))
        else:
            raise PluginException('Could not get collection entries: Collection does not exist.')
        return entries

    def collection_dict(self, value, collection_id) -> dict[str,Any]:
        """ Return the collection dictionary containing the collection id """
        items = []
        if collection_id == 'watchlater' or collection_id == 'favorites':
            items = value['builtinCollections'][collection_id]
        else:
            if collection_id in value['unpublishedCollections']:
                items = value['unpublishedCollections'][collection_id]
            else:
                raise PluginException(f'Could not find the collection dictionary for collection \'{collection_id}\'')
        return items

    def collection_new(self, name: str, type: str='playlist') -> str:
        """ Create a new collection """
        collections = self.collections_list()
        for collection in collections:
            if name == collection[0]:
                return collection[1]

        prefs = self.preferences_get()
        value = prefs[self.preferences_key()]['value']
        collections = value['unpublishedCollections']
        id = str(uuid.uuid4())
        collections[id] = {
            'id' : id,
            'name' : name,
            'itemCount' : 0,
            'type' : type,
            'items' : [],
            'updatedAt' : int(time.time())
            }
        self.preferences_set(prefs[self.preferences_key()])
        return id

    def collection_delete(self, collection_id: str) -> None:
        """ Delete a collection """
        prefs = self.preferences_get()
        value = prefs[self.preferences_key()]['value']
        collections = value['unpublishedCollections']
        if collection_id in collections:
            del collections[collection_id]
        else:
            raise PluginException('Could not remove collection: Collection not found')
        self.preferences_set(prefs[self.preferences_key()])

    def collection_add_item(self, collection_id: str, uri: str) -> None:
        """ Add an item to a collection """
        prefs = self.preferences_get()
        value = prefs[self.preferences_key()]['value']
        coldict = self.collection_dict(value, collection_id)
        items = coldict['items']

        exists = False
        for item in items:
            if uri in item:
                exists = True
                break

        if not exists:
            items.append(f'lbry://{uri}')
            coldict['itemCount'] = len(items)

        coldict['updatedAt'] = int(time.time())

        self.preferences_set(prefs[self.preferences_key()])

    def collection_remove_item(self, collection_id: str, uri: str) -> None:
        """ Remove an item from a collection """
        prefs = self.preferences_get()
        value = prefs[self.preferences_key()]['value']
        coldict = self.collection_dict(value, collection_id)
        items = coldict['items']

        for item in items:
            if uri in item:
                items.remove(item)
                coldict['itemCount'] = len(items)
                break

        coldict['updatedAt'] = int(time.time())

        self.preferences_set(prefs[self.preferences_key()])

    def wallet_balance(self) -> LbryPayload:
        """ Return wallet balance """
        return self.lbrynet_rpc('wallet_balance')

    def support(self, amount: float, channel_id: str, claim_id: str) -> LbryPayload:
        """ Support a claim """
        data: LbryPayload = {
            'amount':  amount,
            'claim_id': claim_id,
            'channel_id': channel_id,
            'tip': amount,
            'blocking': True
        }
        return self.lbrynet_rpc('support_create', data)

    def purchase_list(self, claim_id) -> LbryPayload:
        """ Return purchase list for a claim """
        params: LbryPayload = {'claim_id': claim_id}
        return self.lbrynet_rpc('purchase_list', params)

    def claim_reaction_list(self, claim_ids: str) -> LbryPayload:
        """ List claim reactions """
        if str == '':
            raise PluginException("Cannot get claim reaction list: No claims provided.")

        params: LbryPayload = {
                'claim_ids': claim_ids,
                }
        return self.odysee_rpc('/reaction/list', params).json()

    def preferences_get(self) -> LbryPayload:
        """ Return preferences """
        prefs_key = self.preferences_key()
        prefs = self.lbrynet_rpc('preference_get', {'key': prefs_key })

        if self.local: # Don't do this for Odysee to avoid potential account preferences corruption
            # Returns None if key does not exist
            if not prefs:
                prefs = {}

            if prefs_key not in prefs:
                prefs[prefs_key] = {}

            prefsval = prefs[prefs_key]

            update_prefs = False
            if 'type' not in prefsval:
                prefsval['type'] = 'object'
                update_prefs = True

            if 'value' not in prefsval:
                prefsval['value'] = { 'following' : [], 'subscriptions': [] }
                update_prefs = True

            value = prefsval['value']
            if 'builtinCollections' not in value:
                value['builtinCollections'] = {}
                update_prefs = True

            if 'favorites' not in value['builtinCollections']:
                value['builtinCollections']['favorites'] = {
                        'id': 'favorites',
                        'items' : [],
                        'name' : 'Favorites',
                        'type' : 'playlist',
                        'updatedAt' : int(time.time())
                        }
            if 'watchlater' not in value['builtinCollections']:
                value['builtinCollections']['watchlater'] = {
                        'id': 'watchlater',
                        'items' : [],
                        'name' : 'Watch Later',
                        'type' : 'playlist',
                        'updatedAt' : int(time.time())
                        }

            if 'unpublishedCollections' not in value:
                value['unpublishedCollections'] = {}

            if 'editedCollections' not in value:
                value['editedCollections'] = {}

            if 'updatedCollections' not in value:
                value['updatedCollections'] = {}

            if update_prefs:
                self.preferences_set(prefsval)

        return prefs

    def preferences_set(self, value) -> LbryPayload:
        """ Set preferences """
        return self.lbrynet_rpc('preference_set', {'key': self.preferences_key(), 'value': value})

    def lbrynet_rpc(self, method, params={}, extra_json_vals: LbryPayload={}, extra_headers: Headers={}, lbry_api_url_override: Union[str,None]=None) -> LbryPayload:
        """ Make a remote procedure call to the lbrynet server """
        json = extra_json_vals
        json['method'] = method
        json['params'] = params
        result = requests.post(lbry_api_url_override if lbry_api_url_override else self.lbry_api_url, headers=self.headers|extra_headers, json=json)
        result.raise_for_status()
        rjson: LbryPayload = result.json()
        if 'error' in rjson:
            if rjson['error']['code'] == -32500:
                raise UserChannelDoesNotExist(rjson['error']['message'])
            else:
                raise PluginException(rjson['error']['message'])
        return rjson['result']

    def odysee_comment_rpc(self, method, params={}) -> LbryPayload:
        """ Make a remote procedure call to the Odysee comment server """
        extra_headers = {'content-type' : 'application/json'}
        extra_json_vals = { 'jsonrpc' : '2.0', 'id' : '1' }
        return self.lbrynet_rpc(method, params, lbry_api_url_override=Lbry.ODYSEE_COMMENT_API_URL, extra_json_vals=extra_json_vals, extra_headers=extra_headers)

    def odysee_rpc(self, path: str, params: LbryPayload, headers: Union[Headers,None]=None, raise_exception=True) -> requests.Response:
        """ Issue a post request Odysee """
        result = requests.post(f'https://{Lbry.ODYSEE_API_HOSTNAME}{path}', headers=headers if headers else self.odysee_request_headers(), data=params)
        if raise_exception:
            try:
                result.raise_for_status()
            except Exception as e:
                raise PluginException(str(e))
        return result

    def preferences_key(self) -> str:
        """ Return the preference key. All data is stored under this key. """
        return 'local' if self.local else 'shared'

    def sign(self, data)-> LbryPayload:
        if self.user_channel_id:
            return self.channel_sign(self.user_channel_id, data)
        else:
            raise PluginException('Could not sign: No user channel set')

    def sign_params(self, data, params) -> None:
        res = self.sign(data)
        params['signature'] = res['signature']
        params['signing_ts'] = res['signing_ts']

    def channel_sign(self, channel_id, data) -> LbryPayload:
        """ Sign data with owned channel id """
        def to_hex(s) -> str:
            s = unquote_plus(quote_plus(s,encoding='utf-8'))
            res = '';
            for c in s:
                s = format(ord(c), 'x')
                if len(s) == 1:
                    s = '0' + s;
                res += s
            return res
        return self.lbrynet_rpc('channel_sign', params={'channel_id': channel_id, 'hexdata': to_hex(data)})

    def odysee_request_headers(self) -> Headers:
        """ Init the Odysee API request headers """
        return {
                'authority' : Lbry.ODYSEE_API_HOSTNAME,
                'accept' : '*/*',
                'content-type': 'application/x-www-form-urlencoded',
                'origin': 'https://odysee.com',
                'referer': 'https://odysee.com/',
                'sec-fetch-dest' : 'empty',
                'sec-fetch-mode' : 'cors',
                'sec-fetch-site' : 'same-site',
                'user-agent' : 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36',
                }
