# -*- coding: utf-8 -*-
from __future__ import absolute_import

import routing
import os
import platform
import requests
import time
from urllib.parse import quote,unquote,quote_plus,unquote_plus
import xbmc
import xbmcaddon
from xbmcgui import ListItem, Dialog, NOTIFICATION_ERROR, Window, WindowXMLDialog
from xbmcplugin import addDirectoryItem, addDirectoryItems, endOfDirectory, setContent, setResolvedUrl

from resources.lib.local import *
from resources.lib.exception import *
from resources.lib.lbry import *
from resources.lib.odysee import *
from resources.lib.comment_window import CommentWindow

def load_odysee(odysee: Odysee):
    with open_file('odysee_login', 'r') as f:
        odysee.loadf(f)

def save_odysee(odysee: Odysee):
    with open_file('odysee_login', 'w') as f:
        odysee.savef(f)

def invalidate_subscription_cache() -> None:
    delete_file('subscription_cache') # invalidate cache

ADDON = xbmcaddon.Addon()
tr = ADDON.getLocalizedString

items_per_page = ADDON.getSettingInt('items_per_page')

plugin = routing.Plugin()
ph = plugin.handle
setContent(ph, 'videos')
dialog = Dialog()

lbry_api_url = unquote(ADDON.getSetting('lbry_api_url'))

invalidate_subscription_cache()

# Upgrade older user settings
if lbry_api_url == '':
    ADDON.setSettingBool('use_odysee', True)

# user_channel_{name,id} now separated for Odysee LBRY proxy and self-hosted proxy server.
try:
    legacy_user_channel_name = ADDON.getSettingString('user_channel_name')
    if legacy_user_channel_name != '':
        ADDON.setSettingString('lbry_user_channel_name', legacy_user_channel_name)
        ADDON.setSettingString('user_channel_name', '')

    legacy_user_channel_id = ADDON.getSettingString('user_channel_id')
    if legacy_user_channel_id != '':
        ADDON.setSettingString('lbry_user_channel_id', legacy_user_channel_id)
        ADDON.setSettingString('user_channel_id', '')
except:
    pass

user_channel = get_user_channel()
if ADDON.getSettingBool('use_odysee'):
    lbry = Odysee(user_channel_name=user_channel[0], user_channel_id=user_channel[1])

    try:
        load_odysee(lbry)
    except Exception as e:
        lbry.new()
        save_odysee(lbry)

else:
    lbry = Lbry(lbry_api_url, user_channel_name=user_channel[0],
                user_channel_id=user_channel[1])

def state_exists() -> bool:
    """ Return true if the lbrynet server state exists and can be changed. True
        unless signed out of Odysee.
    """
    if isinstance(lbry, Odysee):
        return lbry.signed_in
    return True

def follows() -> list[Channel]:
    # Odysee does not allow saving subscriptions
    # until signed in so load from user data folder
    if not state_exists():
        return load_local_channel_subs()

    if file_exists('subscription_cache'):
        with open_file('subscription_cache', 'r') as f:
            scache = json.loads(f.read())
            if time.time() - scache['timestamp'] < 30.0:
                subscriptions: list[Channel] = []
                for subscription in scache['subscriptions']:
                    subscriptions.append((subscription[0], subscription[1]))
                return subscriptions

    follows = lbry.subscriptions()
    with open_file('subscription_cache', 'w') as f:
        scache = { 'timestamp': time.time(), 'subscriptions': follows }
        f.write(json.dumps(scache))
    return follows

def subscribe(channel_name: str, claim_id: str) -> None:
    if isinstance(lbry, Odysee):
        if not lbry.signed_in: # Save locally
            channels = load_local_channel_subs()
            channel = (channel_name, claim_id)
            if not channel in channels:
                channels.append(channel)
            save_local_channel_subs(channels)

    invalidate_subscription_cache()
    return lbry.subscribe(channel_name, claim_id)

def unsubscribe(channel_name: str, claim_id: str) -> None:
    if isinstance(lbry, Odysee):
        if not lbry.signed_in:
            channels = load_local_channel_subs()
            channels.remove((channel_name, claim_id))
            save_local_channel_subs(channels)

    invalidate_subscription_cache()
    return lbry.unsubscribe(channel_name, claim_id)

def is_followed(name: str, claim_id: str) -> bool:
    items = follows()
    return True if (name, claim_id) in items else False

def serialize_uri(item) -> str:
    # all uris passed via kodi's routing system must be urlquoted
    if type(item) is dict:
        return quote(item['name'] + '#' + item['claim_id'])
    else:
        return quote(item)

def serialize_comment_uri(item) -> Union[str, None]:
    if 'signing_channel' in item and 'name' in item['signing_channel'] and 'claim_id' in item['signing_channel']:
        signing_channel = item['signing_channel']
        return quote(signing_channel['name'] + '#' + signing_channel['claim_id'] + '#' + item['claim_id'])
    return None

def deserialize_uri(item):
    # all uris passed via kodi's routing system must be urlquoted
    return unquote(item)

def write_trigger_file(filename: str) -> None:
    addon_path = ADDON.getAddonInfo('path')
    trigger_file_path = os.path.join(addon_path, filename)
    with open(trigger_file_path, 'w+') as f:
        pass

def append_follow_toggle_menu_item(menu: list[tuple[str,str]], item: dict) -> None:
    if is_followed(item['name'], item['claim_id']):
        follow_or_unfollow_cb = plugin_unfollow
        follow_or_unfollow_str = tr(30206)
    else:
        follow_or_unfollow_cb = plugin_follow
        follow_or_unfollow_str = tr(30205)

    menu.append((
        follow_or_unfollow_str % item['name'], 'RunPlugin(%s)' % plugin.url_for(follow_or_unfollow_cb, uri=serialize_uri(item))
    ))

def create_channel_listitem(label, channel_info):
    li = ListItem(label)
    if not 'error' in channel_info:
        uri = channel_info['name']+'#'+channel_info['claim_id']
        plot = ''
        if 'title' in channel_info['value'] and channel_info['value']['title'].strip() != '':
            plot = '[B]%s[/B]\n' % channel_info['value']['title']
        else:
            plot = '[B]%s[/B]\n' % channel_info['name']
        if 'description' in channel_info['value']:
            plot = plot + channel_info['value']['description']
        infoLabels = { 'plot': plot }
        li.setInfo('video', infoLabels)

        if 'thumbnail' in channel_info['value'] and 'url' in channel_info['value']['thumbnail']:
            uri = optimized_thumbnail_uri(channel_info['value']['thumbnail']['url'])
            li.setArt({
                'thumb': uri,
                'poster': uri,
                'fanart': uri
            })

        menu = []
        append_follow_toggle_menu_item(menu, channel_info)
        li.addContextMenuItems(menu)
        return li

def to_video_listitem(item, playlist='', channel='', repost=None, show_support=False, is_livestream=False):
    label = item['value']['title'] if 'title' in item['value'] else item['file_name'] if 'file_name' in item else ''

    if 'stream_type' not in item['value']:
        label = f'[COLOR orange]{label}[/COLOR]'

    if show_support:
        label += f" [I][COLOR yellow]({float(item['meta']['support_amount']):.2f} LBC)[/COLOR][/I]"

    li = ListItem(label)
    li.setProperty('IsPlayable', 'true')
    if 'thumbnail' in item['value'] and 'url' in item['value']['thumbnail']:
        uri = optimized_thumbnail_uri(item['value']['thumbnail']['url'])
        li.setArt({
            'thumb': uri,
            'poster': uri,
            'fanart': uri
        })

    infoLabels = {}
    menu = []
    plot = ''
    if 'description' in item['value']:
        plot = item['value']['description']
    if 'author' in item['value']:
        infoLabels['writer'] = item['value']['author']
    elif 'channel_name' in item:
        infoLabels['writer'] = item['channel_name']
    if 'timestamp' in item:
        timestamp = time.localtime(item['timestamp'])
        infoLabels['year'] = timestamp.tm_year
        infoLabels['premiered'] = time.strftime('%Y-%m-%d',timestamp)
    if 'video' in item['value'] and 'duration' in item['value']['video']:
        infoLabels['duration'] = str(item['value']['video']['duration'])

    uri = serialize_comment_uri(item)
    if uri:
        menu.append((
            tr(30238), 'RunPlugin(%s)' % plugin.url_for(plugin_comment_show, uri=uri, is_livestream= '1' if is_livestream else '0')
            ))

    if not is_livestream:
        viduri = item['name'] + '#' + item['claim_id']
        if playlist == '':
            if state_exists():
                menu.append(
                    (tr(30285), 'RunPlugin(%s)' % plugin.url_for(plugin_playlist_add_item, uri=serialize_uri(viduri)))
                    )
        else:
            menu.append(
                (tr(30286), 'RunPlugin(%s)' % plugin.url_for(plugin_playlist_del_item, id=playlist, uri=serialize_uri(viduri)))
                )


        if not isinstance(lbry, Odysee):
            menu.append((
                tr(30208), 'RunPlugin(%s)' % plugin.url_for(claim_download, uri=serialize_uri(item))
            ))

    if 'signing_channel' in item and 'name' in item['signing_channel']:
        ch_name = item['signing_channel']['name']
        ch_claim = item['signing_channel']['claim_id']
        ch_title = ''
        if 'value' in item['signing_channel'] and 'title' in item['signing_channel']['value']:
            ch_title = item['signing_channel']['value']['title']

        plot = '[B]' + (ch_title if ch_title.strip() != '' else ch_name) + '[/B]\n' + plot

        infoLabels['studio'] = ch_name

        if channel == '':
            menu.append((
                tr(30207) % ch_name, 'Container.Update(%s)' % plugin.url_for(lbry_channel, uri=serialize_uri(item['signing_channel']),page=1)
            ))

        append_follow_toggle_menu_item(menu, item['signing_channel'])

    if repost != None:
        if 'signing_channel' in repost and 'name' in repost['signing_channel']:
            plot = (('[COLOR yellow]%s[/COLOR]\n' % tr(30217)) % repost['signing_channel']['name']) + plot
        else:
            plot = ('[COLOR yellow]%s[/COLOR]\n' % tr(30216)) + plot

    infoLabels['plot'] = plot
    li.setInfo('video', infoLabels)
    li.addContextMenuItems(menu)

    return li

def result_to_itemlist(result, playlist='', channel='', show_support=False, include_livestreams=False) -> list:
    if include_livestreams:
        livestream_json = get_all_livestreams_json()
        if 'data' not in livestream_json:
            include_livestreams = False

    nsfw = ADDON.getSettingBool('nsfw')
    items = []
    for item in result:
        if not 'value_type' in item:
            continue
        if item['value_type'] == 'stream':
            if 'tags' in item['value']:
                # nsfw?
                if 'mature' in item['value']['tags'] and not nsfw:
                    continue

                # Unsupported Odysee members-only content. Odysee account integration required to play stream.
                if 'c:members-only' in item['value']['tags']:
                    continue

            if 'stream_type' in item['value'] and item['value']['stream_type'] == 'video':
                li = to_video_listitem(item, playlist, channel, show_support=show_support)
                url = plugin.url_for(claim_play, uri=serialize_uri(item))
                items.append((url, li))
            elif 'stream_type' not in item['value'] and include_livestreams:
                claim_id = item['claim_id']
                if is_livestreaming(livestream_json, claim_id):
                    li = to_video_listitem(item, playlist, channel, show_support=show_support, is_livestream=True)
                    url = plugin.url_for(claim_livestream, uri=serialize_uri(item))
                    items.append((url, li))
        elif item['value_type'] == 'repost' and 'reposted_claim' in item and item['reposted_claim']['value_type'] == 'stream' and item['reposted_claim']['value']['stream_type'] == 'video':
            stream_item = item['reposted_claim']
            # nsfw?
            if 'tags' in stream_item['value']:
                if 'mature' in stream_item['value']['tags'] and not nsfw:
                    continue

            li = to_video_listitem(stream_item, playlist, channel, repost=item, show_support=show_support)
            url = plugin.url_for(claim_play, uri=serialize_uri(stream_item))

            items.append((url, li))
        elif item['value_type'] == 'channel':
            label = '[B]%s[/B] [I]#%s[/I]' % (item['name'], item['claim_id'][0:4])

            if show_support:
                label += f" [I][COLOR yellow]({float(item['meta']['support_amount']):.2f} LBC)[/COLOR][/I]"

            li = create_channel_listitem(label, item)
            url = plugin.url_for(lbry_channel, uri=serialize_uri(item),page=1)
            items.append((url, li, True))
        else:
            xbmc.log('ignored item, value_type=' + item['value_type'])
            xbmc.log('item name=' + item['name'])

    return items

def open_settings():
    xbmc.executebuiltin('Addon.OpenSettings(plugin.video.lbry.sg)')

@plugin.route('/signin_odysee')
def signin_odysee() -> None:
    odysee = Odysee()

    load_odysee(odysee)

    while True:
        email = dialog.input(tr(30291), type=xbmcgui.INPUT_ALPHANUM)
        if email == '':
            return

        try:
            if odysee.user_exists(email):
                break
            else:
                dialog.ok(tr(30287), tr(30290))
        except Exception as e:
            dialog.ok(tr(30287), str(e))
            return

    while True:
        password = dialog.input(tr(30289), type=xbmcgui.INPUT_ALPHANUM, option=xbmcgui.ALPHANUM_HIDE_INPUT)
        if not password:
            open_settings()
            return

        res = odysee.signin(email, password)
        if res[0]:
            break
        else:
            dialog.ok(tr(30287), res[1])

    save_odysee(odysee)
    ADDON.setSettingString('odysee_email', email)

@plugin.route('/signout_odysee')
def signout_odysee() -> None:
    if lbry.signed_in:
        lbry.signout()
    delete_file('odysee_login')
    ADDON.setSettingString('odysee_user_channel_name', '')
    ADDON.setSettingString('odysee_user_channel_id', '')
    ADDON.setSettingString('odysee_email', '')

@plugin.route('/select_user_channel')
def select_user_channel() -> None:
    progressDialog = xbmcgui.DialogProgress()
    progressDialog.create(tr(30231))

    page = 1
    total_pages = 1
    items = []
    while page <= total_pages:
        if progressDialog.iscanceled():
            break

        try:
            result = lbry.owned_channel_list(page=page)
            total_pages = max(result['total_pages'], 1) # Total pages returns 0 if empty
            if 'items' in result:
                items += result['items']
            else:
                break
        except Exception as e:
            progressDialog.close()
            raise e

        page = page + 1
        progressDialog.update(int(100.0*page/total_pages), tr(30220) + ' %s/%s' % (page, total_pages))

    selected_item = None

    if len(items) == 0:
        progressDialog.update(100, tr(30232)) # No owned channels found
        xbmc.sleep(1000)
        progressDialog.close()
        clear_user_channel()
    elif len(items) == 1:
        progressDialog.update(100, tr(30233)) # Found single user
        xbmc.sleep(1000)
        progressDialog.close()

        selected_item = items[0]
    else:
        progressDialog.update(100, tr(30234)) # Multiple users found
        xbmc.sleep(1000)
        progressDialog.close()

        names = []
        for item in items:
            names.append(f"{item['name']}[COLOR green]#{item['claim_id'][:5]}[/COLOR]")

        selected_name_index = dialog.select(tr(30239), names) # Post As

        if selected_name_index >= 0: # If not cancelled
            selected_item = items[selected_name_index]

    if selected_item:
        set_user_channel(selected_item['name'], selected_item['claim_id'])

    open_settings()

@plugin.route('/clear_user_channel')
def _clear_user_channel() -> None:
    clear_user_channel()
    open_settings()

@plugin.route('/install_lbrynet')
def install_lbrynet() -> None:
    def is_platform_supported():
        system = platform.system()
        return system == 'Linux' or system == 'Windows' or system == 'Darwin'

    def is_linux_architecture_supported():
        machine = platform.machine()
        return machine == 'x86_64' or machine == 'aarch64' or machine == 'armv7l'

    if not is_platform_supported():
        xbmcgui.Dialog().ok('Unsupported platform', f'{platform.system()}')
        return
    if platform.system() == 'Linux' and not is_linux_architecture_supported():
        xbmcgui.Dialog().ok('Unsupported architecture', f'{platform.machine()}')
        return

    write_trigger_file('install-lbrynet')

@plugin.route('/uninstall_lbrynet')
def uninstall_lbrynet() -> None:
    write_trigger_file('uninstall-lbrynet')

@plugin.route('/restart_lbrynet')
def restart_lbrynet() -> None:
    write_trigger_file('restart-lbrynet')

@plugin.route('/')
def lbry_root() -> None:
    item = ListItem()
    item.setIsFolder(True)
    item.setProperty('IsPlayable', 'false')

    item.setLabel(tr(30200))
    item.setInfo('video', {'plot': tr(30248)})
    addDirectoryItem(ph, plugin.url_for(plugin_follows), item, True)

    item.setLabel(tr(30218))
    item.setInfo('video', {'plot': tr(30249)})
    addDirectoryItem(ph, plugin.url_for(plugin_recent, page=1, live=0), item, True)

    item.setLabel(tr(30276))
    item.setInfo('video', {'plot': tr(30277)})
    addDirectoryItem(ph, plugin.url_for(plugin_recent, page=1, live=1), item, True)

    # Only enable playlists if are running a lbrynet instance or signed into
    # Odysee. Odysee does not support storing playlists when signed out.
    if state_exists():
        item.setLabel(tr(30210))
        item.setInfo('video', {'plot': tr(30250)})
        addDirectoryItem(ph, plugin.url_for(plugin_playlists), item, True)

    item.setLabel(tr(30202))
    item.setInfo('video', {'plot': tr(30251)})
    addDirectoryItem(ph, plugin.url_for(lbry_new, page=1), item, True)

    item.setLabel(tr(30201))
    item.setInfo('video', {'plot': tr(30252)})
    addDirectoryItem(ph, plugin.url_for(lbry_search, channels_only='0'), item, True)

    endOfDirectory(ph)

@plugin.route('/playlists/')
def plugin_playlists() -> None:
    playlists = lbry.collections_list()

    li = ListItem()
    li.setIsFolder(True)
    li.setProperty('IsPlayable', 'false')
    li.setLabel(tr(30292))
    addDirectoryItem(ph, plugin.url_for(plugin_playlist_new), li, True)
    for playlist in playlists:
        li.setLabel(playlist[0])
        menu = []
        if playlist[1] != 'watchlater' and playlist[1] != 'favorites':
            menu.append(('Delete', 'RunPlugin(%s)' % plugin.url_for(plugin_playlist_delete, id=playlist[1])))
        li.addContextMenuItems(menu)
        addDirectoryItem(ph, plugin.url_for(plugin_playlist_list, quote(playlist[1])), li, True)

    endOfDirectory(ph)

@plugin.route('/playlist/list/<id>/')
def plugin_playlist_list(id: str) -> None:
    uris = lbry.collection_items(id)
    claim_info = lbry.resolve(uris)
    items = []
    for uri in uris:
        items.append(claim_info[uri])
    items = result_to_itemlist(items, playlist=id)
    addDirectoryItems(ph, items, len(items))
    endOfDirectory(ph)

@plugin.route('/playlist/new/')
def plugin_playlist_new() -> None:
    name = dialog.input(tr(30293), type=xbmcgui.INPUT_ALPHANUM)
    if name != '':
        lbry.collection_new(name)
        time.sleep(1)
        xbmc.executebuiltin('Container.Refresh')

@plugin.route('/playlist/del/<id>/')
def plugin_playlist_delete(id: str) -> None:
    lbry.collection_delete(id)
    xbmc.executebuiltin('Container.Refresh')

@plugin.route('/playlist/add_item/<uri>')
def plugin_playlist_add_item(uri: str) -> None:
    uri = deserialize_uri(uri)
    collections = lbry.collections_list()
    selections = []
    for collection in collections:
        selections.append(collection[0])
    indices = dialog.multiselect(tr(30210), selections)
    if indices:
        for idx in indices:
            lbry.collection_add_item(collections[idx][1], uri)
        xbmc.executebuiltin('Container.Refresh')

@plugin.route('/playlist/del_item/<id>/<uri>')
def plugin_playlist_del_item(id: str, uri: str) -> None:
    uri = deserialize_uri(uri)
    id = unquote_plus(id)
    lbry.collection_remove_item(id, uri)
    xbmc.executebuiltin('Container.Refresh')

@plugin.route('/follows')
def plugin_follows():
    li = ListItem()
    li.setIsFolder(True)
    li.setProperty('IsPlayable', 'false')
    li.setLabel(tr(30292))
    addDirectoryItem(ph, plugin.url_for(lbry_search, channels_only='1'), li, True)

    channels = follows()
    resolve_uris = []
    for (name,claim_id) in channels:
        resolve_uris.append(name+'#'+claim_id)
    channel_infos = lbry.resolve(resolve_uris)

    for (name,claim_id) in channels:
        uri = name+'#'+claim_id
        channel_info = channel_infos[uri]
        if 'error' not in channel_info: # Odysee proxy will throw an error with banned channels
            li = create_channel_listitem(name, channel_info)
            addDirectoryItem(ph, plugin.url_for(lbry_channel, uri=serialize_uri(uri), page=1), li, True)

    endOfDirectory(ph)

@plugin.route('/recent/<page>/<live>')
def plugin_recent(page: str, live: str) -> None:
    ipage = int(page)
    channels = follows()
    if len(channels) != 0:
        channel_ids = []
        for (name,claim_id) in channels:
            channel_ids.append(claim_id)
        is_live = live == '1'
        result = lbry.claim_search(channel_ids=channel_ids, page=ipage, page_size=items_per_page, order_by=Lbry.ORDER_BY_RELEASE_TIME, live=is_live, claim_type=['stream','repost'], stream_types=['video'])
        items = result_to_itemlist(result['items'], include_livestreams=True)
        addDirectoryItems(ph, items, result['page_size'])
        if live == '0':
            total_pages = int(result['total_pages'])
            if total_pages > 1 and ipage < total_pages:
                addDirectoryItem(ph, plugin.url_for(plugin_recent, page=ipage+1, live=live), ListItem(tr(30203)), True)
    endOfDirectory(ph)

@plugin.route('/comments/show/<uri>/<is_livestream>')
def plugin_comment_show(uri: str, is_livestream: str):
    params = deserialize_uri(uri).split('#')
    CommentWindow(params[0], params[1], params[2], is_livestream == '1', lbry=lbry, user_channel=get_user_channel())

@plugin.route('/follows/add/<uri>')
def plugin_follow(uri: str) -> None:
    uri = deserialize_uri(uri)
    channel = uri.split('#')
    subscribe(channel[0], channel[1])
    xbmc.executebuiltin('Container.Refresh')

@plugin.route('/follows/del/<uri>')
def plugin_unfollow(uri: str) -> None:
    uri = deserialize_uri(uri)
    channel = uri.split('#')
    unsubscribe(channel[0], channel[1])
    xbmc.executebuiltin('Container.Refresh')

@plugin.route('/new/<page>')
def lbry_new(page: str) -> None:
    ipage = int(page)
    result = lbry.claim_search(page=ipage, page_size=items_per_page, order_by=Lbry.ORDER_BY_TRENDING_GLOBAL, claim_type=['stream'], stream_types=['video'])
    items = result_to_itemlist(result['items'])
    addDirectoryItems(ph, items, result['page_size'])
    total_pages = int(result['total_pages'])
    if total_pages > 1 and ipage < total_pages:
        addDirectoryItem(ph, plugin.url_for(lbry_new, page=ipage+1), ListItem(tr(30203)), True)
    endOfDirectory(ph)

@plugin.route('/channel/<uri>')
def lbry_channel_landing(uri) -> None:
    lbry_channel(uri,'1')

@plugin.route('/channel/<uri>/<page>')
def lbry_channel(uri: str, page: str) -> None:
    uri = deserialize_uri(uri)
    ipage = int(page)
    result = lbry.claim_search(page=ipage, page_size=items_per_page, order_by=Lbry.ORDER_BY_RELEASE_TIME, channel=uri, claim_type=['stream','repost'], stream_types=['video'])
    items = result_to_itemlist(result['items'], channel=uri)
    addDirectoryItems(ph, items, result['page_size'])
    total_pages = int(result['total_pages'])
    if total_pages > 1 and ipage < total_pages:
        addDirectoryItem(ph, plugin.url_for(lbry_channel, uri=serialize_uri(uri), page=ipage+1), ListItem(tr(30203)), True)
    endOfDirectory(ph)

@plugin.route('/search/<channels_only>')
def lbry_search(channels_only: str) -> None:
    query = dialog.input(tr(30209))
    if len(query):
        lbry_search_pager(quote_plus(query), '1', channels_only)
    xbmc.executebuiltin('Container.Update(%s,replace)' % plugin.url_for(lbry_search_pager, query=quote_plus(query), page=1, channels_only=channels_only))

@plugin.route('/search/<query>/<page>/<channels_only>')
def lbry_search_pager(query: str, page: str, channels_only: str) -> None:
    query = unquote_plus(query)
    ipage = int(page)
    if query != '':
        claim_type = ['channel'] if channels_only == '1' else None
        result = lbry.claim_search(text=query, page=ipage, page_size=items_per_page, order_by=Lbry.ORDER_BY_SUPPORT_AMOUNT, claim_type=claim_type)
        items = result_to_itemlist(result['items'], show_support=True)
        addDirectoryItems(ph, items, result['page_size'])
        total_pages = int(result['total_pages'])
        if total_pages > 1 and ipage < total_pages:
            addDirectoryItem(ph, plugin.url_for(lbry_search_pager, query=quote_plus(query), page=ipage+1, channels_only=channels_only), ListItem(tr(30203)), True)
        endOfDirectory(ph)
    else:
        endOfDirectory(ph, False)

def user_payment_confirmed(claim_info: dict['str',Any]) -> bool:
    # paid for claim already?
    purchase_info = lbry.purchase_list(claim_info['claim_id'])
    if len(purchase_info['items']) > 0:
        return True

    account_list = lbry.account_list()
    balance = 0.0
    for account in account_list['items']:
        if account['is_default']:
            balance = float(str(account['satoshis'])[:-6]) / float(100)
    dtext = tr(30214) % (float(claim_info['value']['fee']['amount']), str(claim_info['value']['fee']['currency']))
    dtext = dtext + '\n\n' + tr(30215) % (balance, str(claim_info['value']['fee']['currency']))
    return dialog.yesno(tr(30204), dtext)

@plugin.route('/livestream/<uri>')
def claim_livestream(uri: str) -> None:
    uri = deserialize_uri(uri)
    claim_info = lbry.resolve(uri)[uri]
    (url,li) = result_to_itemlist([claim_info], include_livestreams=True)[0]
    claim_id = claim_info['signing_channel']['claim_id']
    url = f'https://cloud.odysee.live/content/{claim_id}/master.m3u8|referer=https://odysee.com/'
    li.setPath(url)
    setResolvedUrl(ph, True, li)

@plugin.route('/play/<uri>')
def claim_play(uri: str) -> None:
    uri = deserialize_uri(uri)

    claim_info = lbry.resolve(uri)[uri]
    if 'error' in claim_info:
        dialog.notification(tr(30102), claim_info['error']['name'], NOTIFICATION_ERROR)
        return

    if 'fee' in claim_info['value']:
        if claim_info['value']['fee']['currency'] != 'LBC':
            dialog.notification(tr(30204), tr(30103), NOTIFICATION_ERROR)
            return

        if not user_payment_confirmed(claim_info):
            return

    lbry_streaming_url_override = None
    if ADDON.getSettingBool('stream_from_odysee_proxy'):
        lbry_streaming_url_override = Odysee.PROXY_URL
        using_proxy = True
    else:
        using_proxy = lbry.lbry_api_url == Odysee.PROXY_URL

    result = lbry.get(uri, False, lbry_streaming_url_override=lbry_streaming_url_override)
    stream_url = result['streaming_url'].replace('0.0.0.0','127.0.0.1')

    # Fixes an issue with some videos not playing through the Odysee proxy
    if using_proxy:
        stream_url += '|Referer=https://odysee.com/'

    (url,li) = result_to_itemlist([claim_info])[0]
    li.setPath(stream_url)
    setResolvedUrl(ph, True, li)

@plugin.route('/download/<uri>')
def claim_download(uri: str) -> None:
    uri = deserialize_uri(uri)

    claim_info = lbry.resolve(uri)[uri]
    if 'error' in claim_info:
        dialog.notification(tr(30102), claim_info['error']['name'], NOTIFICATION_ERROR)
        return

    if 'fee' in claim_info['value']:
        if claim_info['value']['fee']['currency'] != 'LBC':
            dialog.notification(tr(30204), tr(30103), NOTIFICATION_ERROR)
            return

        if not user_payment_confirmed(claim_info):
            return

    lbry.get(uri, True)

@plugin.route('/export')
def export_data() -> None:
    directory = dialog.browse(3, tr(30301), '')
    if directory != '':
        filename = dialog.input(tr(30300), 'lbry_exports.json')
        if filename != '':
            path = directory + filename
            if xbmcvfs.exists(path):
                if not dialog.yesno(tr(30296), tr(30299)):
                    return
            with xbmcvfs.File(path, 'w') as f:
                signed_out = ADDON.getSettingBool('use_odysee') and not lbry.signed_in

                if signed_out:
                    subs = load_local_channel_subs()
                    playlists = []
                else:
                    subs = lbry.subscriptions()
                    collections = lbry.collections_list()
                    playlists = []
                    for collection in collections:
                        items = lbry.collection_items(collection[1])
                        playlists.append({ 'name': collection[0], 'items': items })

                output = {
                          'file_type' : 'lbry_export',
                          'subscriptions' : subs,
                          'playlists' : playlists
                          }
                f.write(json.dumps(output))
                dialog.ok(tr(30298), tr(30304) + ' ' + path)

@plugin.route('/import')
def import_data() -> None:
    path = dialog.browse(1, tr(30305), "files")
    if path != '':
        with xbmcvfs.File(path, 'r') as f:
            def errdiag():
                dialog.ok(tr(30100), tr(30306))

            try:
                js = json.loads(f.read())
            except Exception:
                errdiag()
                return
            if 'file_type' not in js or js['file_type'] != 'lbry_export':
                errdiag()
                return

            signed_out = ADDON.getSettingBool('use_odysee') and not lbry.signed_in

            pdiag = xbmcgui.DialogProgress()
            pdiag.create(tr(30307))
            nsubs = len(js['subscriptions'])
            idx = 0
            for subscription in js['subscriptions']:
                xbmc.log(f'Importing subscription: {subscription[0]}#{subscription[1]}')
                idx += 1
                if pdiag.iscanceled():
                    pdiag.close()
                    return
                success = True
                if signed_out:
                    subs = load_local_channel_subs()
                    sub = (subscription[0], subscription[1])
                    if not sub in subs:
                        subs.append(sub)
                    save_local_channel_subs(subs)
                else:
                    try:
                        lbry.subscribe(subscription[0], subscription[1])
                    except Exception as e:
                        xbmc.log(f'Failed to import subscription \'{subscription[0]}#{subscription[1]}\': {str(e)}')
                        success = False

                success_str = f'[COLOR green]{tr(30308)} {subscription[0]}#{subscription[1]}[/COLOR]'
                failed_str = f'[COLOR red]{tr(30309)} {subscription[0]}#{subscription[1]}[/COLOR]'
                pdiag.update(int(idx/nsubs*100), success_str if success else failed_str)
            pdiag.close()

            if not signed_out:
                pdiag = xbmcgui.DialogProgress()
                pdiag.create('Playlists')
                nplaylists = len(js['playlists'])
                idx = 0
                for playlist in js['playlists']:
                    name = playlist['name']
                    xbmc.log(f'Importing playlist: {name}')
                    idx += 1
                    if pdiag.iscanceled():
                        pdiag.close()
                        return
                    success = True
                    try:
                        id = lbry.collection_new(playlist['name'])
                        for item in playlist['items']:
                            lbry.collection_add_item(id, item)
                    except Exception as e:
                        xbmc.log(f'Failed to import playlist: {name}')
                        success = False

                    success_str = f'[COLOR green]{tr(30310)} {name}[/COLOR]'
                    failed_str = f'[COLOR red]{tr(30311)} {name}[/COLOR]'
                    pdiag.update(int(idx/nplaylists*100), success_str if success else failed_str)
                pdiag.close()

def run() -> None:
    try:
        plugin.run()
    except PluginException as e:
        dialog.notification(tr(30102), str(e), NOTIFICATION_ERROR)
        xbmc.log("PluginException: " + str(e))
    except requests.exceptions.ConnectionError as e:
        dialog.notification(tr(30105), tr(30106), NOTIFICATION_ERROR)
        xbmc.log("PluginException: " + str(e))
    except requests.exceptions.HTTPError as e:
        dialog.notification(tr(30101), str(e), NOTIFICATION_ERROR)
        xbmc.log("PluginException: " + str(e))
    except Exception as e:
        xbmc.log('Exception:' + str(e))
        raise e
