#from local import call_rpc
import requests
import json
from typing import Union

from resources.lib.lbry import *

class Odysee(Lbry):
    API_HOSTNAME = 'api.odysee.com'
    PROXY_URL = 'https://api.na-backend.odysee.com/api/v1/proxy'
    ODYSEE_APP_ID = 'odyseecom692EAWhtoqDuAfQ6KHMXxFxt8tkhmt7sfprEMHWKjy5hf6PwZcHDV542V'

    def __init__(self, user_channel_name='', user_channel_id=''):
        super().__init__(Odysee.PROXY_URL, local=False, user_channel_name=user_channel_name, user_channel_id=user_channel_id)
        self.signed_in: bool = False
        self.email: Union[str,None] = None

    def new(self) -> None:
        """ Create a new user. """
        data = {
            'auth_token': '',
            'language': 'en',
            'app_id': Odysee.ODYSEE_APP_ID
        }
        result = requests.get(self.url('/user/new'), data=data)
        try:
            result.raise_for_status();
        except Exception as e:
            raise PluginException(str(e))
        rjson = result.json()
        if rjson and rjson[ 'success' ]:
            self.auth_token = rjson[ 'data' ][ 'auth_token' ]
            self.init_request_headers(self.auth_token)

    def load(self, path: str) -> None:
        """ Load from file """
        try:
            with open(path, 'r') as f:
                self.loadf(f)
        except FileNotFoundError:
            pass

    def loadf(self, f) -> None:
        """ Load from file handle """
        js = json.loads(f.read())
        if 'auth_token' in js:
            self.auth_token = js['auth_token']
        if 'email' in js:
            self.email = js['email']
            self.signed_in = True
        else:
            if self.email:
                self.email = None
            self.signed_in = False
        self.init_request_headers(self.auth_token)

    def save(self, path: str) -> None:
        """ Save to file """
        with open(path, 'w') as f:
            self.savef(f)

    def savef(self, f) -> None:
        """ Save from file handle """
        dict = {
                'auth_token' : self.auth_token,
                }
        if self.signed_in:
            dict['email'] = self.email
        f.write(json.dumps(dict))

    def user_exists(self, email: str) -> Any:
        """ Checks if the user exists based upon email """
        data = {
            'auth_token': self.auth_token,
            'email': email,
        }
        result = requests.post(self.url('/user/exists'), data=data).json()
        return result and result['success']

    def signin(self, email: str, password: str) -> tuple[bool,str]:
        """ Attempt to sign into Odysee """
        if self.user_exists(email):
            data = {
                    'auth_token': self.auth_token,
                    'email': email,
                    'password': password,
                    }
            result = requests.post(self.url('/user/signin'), data=data)
            if not result.ok:
                try:
                    if result.json():
                        return (False, result.json()['error'])
                except:
                    try:
                        result.raise_for_status();
                    except Exception as e:
                        return (False, str(e))

            rjson = result.json()
            self.signed_in = rjson and rjson['success']
            if self.signed_in:
                self.email = email
            return (self.signed_in, rjson['error'])

        return (False, 'user does not exist')

    def signout(self) -> None:
        """ Sign out """
        data = {
                'auth_token': self.auth_token,
                }
        result = requests.post(self.url('/user/signout'), data=data)
        try:
            result.raise_for_status();
        except Exception as e:
            raise PluginException(str(e))
        self.signed_in = False
        self.email = ''

    def me(self) -> dict[str,Any]:
        """ Get information about the user """
        data = { 'auth_token': self.auth_token }
        return requests.post(self.url('/user/me'), data=data).json()

    def subscribe(self, channel_name: str, claim_id: str) -> None:
        """ Subscribe to a channel """
        if self.signed_in:
            if channel_name:
                data = {
                        'auth_token': self.auth_token,
                        'channel_name': channel_name,
                        'claim_id': claim_id,
                        'notifications_disabled': 'true',
                        }
                result = requests.post(self.url('/subscription/new'), headers=self.odysee_request_headers(), data=data)
                try:
                    result.raise_for_status();
                except Exception as e:
                    raise PluginException(str(e))
                rjson = result.json()
                if rjson and rjson['success']:
                    super().subscribe(channel_name, claim_id)
        else:
            pass

    def unsubscribe(self, channel_name: str, claim_id: str) -> None:
        """ Unsubscribe to a channel """
        if self.signed_in:
            if claim_id:
                data = {
                        'auth_token': self.auth_token,
                        'claim_id': claim_id,
                        }
                result = requests.post(self.url('/subscription/delete'), headers=self.odysee_request_headers(), data=data)
                try:
                    result.raise_for_status();
                except Exception as e:
                    raise PluginException(str(e))
                rjson = result.json()
                if rjson and rjson['success']:
                    super().unsubscribe(channel_name, claim_id)
        else:
            pass

    def url(self, path: str) -> str:
        """ Return the full url including the path """
        return f'https://{Odysee.API_HOSTNAME}{path}'

    def init_request_headers(self, auth_token) -> None:
        self.headers = {'content-type' : 'application/json', 'x-lbry-auth-token' : auth_token }

    def odysee_request_headers(self) -> Headers:
        return {
                'authority' : Odysee.API_HOSTNAME,
                'accept' : '*/*',
                'content-type': 'application/x-www-form-urlencoded',
                'origin': 'https://odysee.com',
                'referer': 'https://odysee.com/',
                'sec-fetch-dest' : 'empty',
                'sec-fetch-mode' : 'cors',
                'sec-fetch-site' : 'same-site',
                'user-agent' : 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36',
                }
