import os
import platform
import requests
import socket
import stat
import subprocess
import time
import urllib.parse
import urllib.request
import xbmc
import xbmcaddon
import xbmcgui
import xbmcvfs
import zipfile

from resources.lib.local import *

dl_links = {
    'Linux-x86_64' : ('https://github.com/lbryio/lbry-sdk/releases/download/v0.106.0/lbrynet-linux.zip', True),
    'Linux-aarch64' : ('https://slavegrid.com/kodi/lbrynet/v0.106.0/lbrynet-linux-aarch64.zip', False),
    'Windows' : ('https://github.com/lbryio/lbry-sdk/releases/download/v0.106.0/lbrynet-windows.zip', True),
    'Darwin' : ('https://github.com/lbryio/lbry-sdk/releases/download/v0.106.0/lbrynet-mac.zip', True)
}

clean_cache_interval = 300 # Check every 5 minutes
predownload_interval = 300 # Check every 5 minutes
local_server_url = 'http://localhost:5279'

addon = xbmcaddon.Addon()
addon_path = addon.getAddonInfo('path')
tr = addon.getLocalizedString
lbrynet_process = None
lbrynet_path = os.path.join(addon_path, 'lbrynet')

def is_windows():
    return platform.system() == 'Windows'

def is_linux():
    return platform.system() == 'Linux'

if is_windows():
    lbrynet_path += '.exe'

profile_path = xbmcvfs.translatePath(addon.getAddonInfo('profile'))
daemon_config_path = os.path.join(profile_path, 'daemon_settings.yml')

def get_ip():
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.settimeout(0)
    try:
        s.connect(('10.255.255.255', 1))
        ip = s.getsockname()[0]
    except Exception:
        ip = '127.0.0.1'
    finally:
        s.close()
    return ip

def get_default_config():
    return \
        f"data_dir: {profile_path}lbrynet\n" \
        f"wallet_dir: {profile_path}lbryum\n" \
        f"download_dir: {profile_path}download\n"

def refresh_configuration():
    config = ""
    with open(daemon_config_path, 'r') as f:
        for line in f:
            line.lstrip()
            if 'network_interface:' not in line and \
               'api:' not in line and \
               'streaming_server' not in line and \
               'use_upnp' not in line and \
               'share_usage_data' not in line: \
                config += line

    addon = xbmcaddon.Addon()

    if not addon.getSettingBool('allow_seeding'):
        config += 'network_interface: 127.0.0.1\n'

    if not addon.getSettingBool('enable_upnp'):
        config += 'use_upnp: false\n'

    if not addon.getSettingBool('share_usage_data'):
        config += 'share_usage_data: false\n'

    if xbmcaddon.Addon().getSettingBool('listen_to_external_connections'):
        # wait for the network interface to come up so we can grab the internal ip
        tries = 6
        while tries > 0:
            ip = get_ip()
            if ip == '127.0.0.1':
                xbmc.sleep(5)
                tries -= 1
            else:
                break

        config += f'api: 0.0.0.0:5279\n'
        config += f'streaming_server: {ip}:5280\n' # 0.0.0.0 causes the server to return links to localhost

    with open(daemon_config_path, 'w') as f:
        f.write(config)

    xbmc.log('Refreshed configuration')

def start_lbrynet():
    global lbrynet_process
    installed = False
    if os.path.exists(lbrynet_path):
        if not lbrynet_process:
            refresh_configuration()
            profile_path = xbmcvfs.translatePath(xbmcaddon.Addon().getAddonInfo('profile'))
            lbrynet_process = subprocess.Popen([addon_path + 'lbrynet', 'start', f'--config={daemon_config_path}'])
        installed = True
    else:
        installed = False

    xbmcaddon.Addon().setSettingBool('is_daemon_installed', installed)

def stop_lbrynet():
    global lbrynet_process
    if lbrynet_process:
        if is_windows():
            os.system('taskkill /im lbrynet.exe /f')
            xbmc.sleep(5000)
            lbrynet_process = None
            return True
        else:
            lbrynet_process.terminate()
            try:
                lbrynet_process.wait(timeout=30)
                lbrynet_process = None
                return True
            except TimeoutExpired:
                return False

    return True

def restart_lbrynet():
    stop_lbrynet()
    start_lbrynet()
    return xbmcgui.Dialog().ok('LBRY', tr(30253)) # Restarted lbrynet

def should_replace_config():
    return xbmcgui.Dialog().yesno('LBRY', tr(30254)) # "Replace the existing daemon configuration with the default configuration?"

last_chunk = 0
t_last = 0.0
bitrate = 0.0

def download_url(url, progress_dialog):
    global last_chunk
    global t_last
    global bitrate
    last_chunk = 0
    t_last = time.time()
    t_cur = t_last
    bittrate = 0

    def dl_report_hook(chunk, block_size, total_size):
        global last_chunk
        global t_last
        global bitrate

        if progress_dialog.iscanceled():
            raise Exception()

        t_cur = time.time()
        t_delta = t_cur - t_last
        if t_delta > 1.0:
            sz_delta = (chunk-last_chunk)*block_size
            bitrate = sz_delta/t_delta
            t_last = t_cur
            last_chunk = chunk

        o = urllib.parse.urlparse(url)
        progress_dialog.update(100*chunk*block_size//total_size, f'{tr(30255)} {o.hostname}\n\n{chunk*block_size/(1000*1000):.2f}/{total_size/(1000*1000):.2f} MB @ {bitrate/(1000*1000):.2f} MB/s')

    return urllib.request.urlretrieve(url, reporthook=dl_report_hook)

def install_lbrynet():
    xbmc.log('Installing lbrynet binary')
    system = platform.system()
    if is_linux():
        machine = platform.machine()
        (dl_link,is_official) = dl_links[f"{system}-{machine}"]
    else:
        (dl_link,is_official) = dl_links[system]

    if not is_official:
        if not xbmcgui.Dialog().yesno('LBRY', tr(30256)):
            return

    progress_dialog = xbmcgui.DialogProgress()
    progress_dialog.create('LBRY', tr(30257))
    xbmc.sleep(1000)

    try:
        (dl_filename, headers) = download_url(dl_link, progress_dialog)

        with zipfile.ZipFile(dl_filename, 'r') as zip_ref:
            if stop_lbrynet():
                zip_ref.extractall(addon_path)
                st = os.stat(lbrynet_path)
                os.chmod(lbrynet_path, st.st_mode | stat.S_IEXEC)

                profile_path = xbmcvfs.translatePath(xbmcaddon.Addon().getAddonInfo('profile'))
                if not os.path.exists(daemon_config_path) or should_replace_config():
                    with open(daemon_config_path, 'w+') as f:
                        f.write(get_default_config())

                progress_dialog.update(100, tr(30258))
                start_lbrynet()
                xbmcaddon.Addon().setSetting('lbry_api_url', local_server_url)

                # Give 10 seconds to spin up so the users aren't given an exception about unloaded components when they try to play.
                xbmc.sleep(10000)

                # Skip this message if the user hits cancel as its only there for feedback.
                if not progress_dialog.iscanceled():
                    progress_dialog.update(100, tr(30259))
                    xbmc.sleep(4000)

                xbmc.log('Installed lbrynet binary')
            else:
                progress_dialog.update(100, tr(30260))
                xbmc.log('Failed to install lbrynet binary')
    except urllib.request.URLError as e:
        progress_dialog.update(100, f'{tr(30261)}: "{str(e)}"')
        xbmc.sleep(5000)
        xbmc.log('Failed to install lbrynet binary')
    except:
        progress_dialog.update(100, tr(30262))
        xbmc.sleep(2000)
        xbmc.log('Failed to install lbrynet binary')

    progress_dialog.close()
    urllib.request.urlcleanup()

def uninstall_lbrynet():
    progress_dialog = xbmcgui.DialogProgress()
    progress_dialog.create('LBRY', tr(30263))
    xbmc.sleep(500)
    stop_lbrynet()
    if os.path.exists(lbrynet_path):
        os.remove(lbrynet_path)
    addon = xbmcaddon.Addon()
    addon.setSettingBool('is_daemon_installed', False)
    addon.setSetting('lbry_api_url', "https://api.lbry.tv/api/v1/proxy")
    progress_dialog.update(100, tr(30264))
    xbmc.sleep(2000)
    progress_dialog.close()

def process_trigger_file(filename, hook):
    trigger_file_path = os.path.join(addon_path, filename)
    if os.path.exists(trigger_file_path):
        if hook:
            hook()
        os.remove(trigger_file_path)

def get_size(path):
    total_size = 0
    for dirpath, dirnames, filenames in os.walk(path):
        for f in filenames:
            fp = os.path.join(dirpath, f)
            if not os.path.islink(fp):
                total_size += os.path.getsize(fp)
    return total_size

def blob_cache_size():
    blob_path = os.path.join(xbmcvfs.translatePath(xbmcaddon.Addon().getAddonInfo('profile')), 'lbrynet/blobfiles')
    return get_size(blob_path)

def clean_cache():
    xbmc.log('Cleaning cache')
    try:
        json = { 'method' : 'file_list', 'params' : {} }
        result = requests.post('http://localhost:5279', json=json)
        result.raise_for_status()
        rjson = result.json()
        xbmc.log(f'Cache size: {blob_cache_size()/(1024**2):.2f}MiB')
        delete_item_idx = 0
        while blob_cache_size() > xbmcaddon.Addon().getSettingInt('max_cache_size')*(1024**3):
            if 'result' not in rjson or 'items' not in rjson['result'] or delete_item_idx >= len(rjson['result']['items']):
                break

            item_to_delete = rjson['result']['items'][delete_item_idx]
            json = { 'method' : 'file_delete', 'params' : { 'claim_id' : item_to_delete['claim_id'], 'delete_from_download_dir' : True } }
            requests.post(local_server_url, json=json)
            xbmc.log(f"Deleted blobs for file {item_to_delete['metadata']['title']} (Cache size: {blob_cache_size()/(1024**2):.2f}MiB)")
            delete_item_idx += 1
    except Exception as e:
        xbmc.log(f'Failed to clean cache: {str(e)}')
        pass
    xbmc.log('Finished cleaning cache')

def predownload():
    xbmc.log('Predownloading recent videos')
    try:
        channels = load_channel_subs()
        if len(channels) != 0:
            channel_ids = []
            for (name,claim_id) in channels:
                channel_ids.append(claim_id)
            preload_video_count = xbmcaddon.Addon().getSettingInt('preload_recent_video_count')
            json = {
                'method': 'claim_search',
                'params': { 'page': 1, 'page_size': preload_video_count, 'order_by': 'release_time', 'channel_ids': channel_ids }
                }
            result = requests.post(local_server_url, json=json)
            result.raise_for_status()
            rjson = result.json()
            uri_list = []
            for item in rjson['result']['items']:
                uri_list.append(f"{item['name']}#{item['claim_id']}")
            uri_list = reversed(uri_list)
            for uri in uri_list:
                json = {
                    'method': 'get',
                    'params': {'uri': uri, 'save_file': True }
                    }
                requests.post(local_server_url, json=json)
    except Exception as e:
        xbmc.log(f'Failed to predownload: {str(e)}')
    xbmc.log('Finished predownloading recent videos')

class Monitor(xbmc.Monitor):
    def __init__(self, *args, **kwargs):
        xbmc.Monitor.__init__(self)

        # Restart server if any of these settings are changed
        self.restart_keys = [ 'allow_seeding', 'listen_to_external_connections', 'enable_upnp', 'share_usage_data' ]
        self.last_restart_key_values = {}
        for key in self.restart_keys:
            self.last_restart_key_values[key] = xbmcaddon.Addon().getSettingBool(key)

    def onSettingsChanged(self):
        if not os.path.exists(lbrynet_path):
            xmbc.log(lbrynet_path)
            return

        xbmc.log(str(self.last_restart_key_values))
        xbmc.log(str(self.restart_keys))
        trigger_restart = False
        for key in self.restart_keys:
            val = xbmcaddon.Addon().getSettingBool(key)
            xbmc.log(str(val))
            if self.last_restart_key_values[key] != val:
                trigger_restart = True
                self.last_restart_key_values[key] = val

        if trigger_restart:
            restart_lbrynet()

if __name__ == '__main__':
    start_lbrynet()

    time_since_last_cache_clean = time.time()
    time_since_last_predownload = time_since_last_cache_clean

    monitor = Monitor()
    while not monitor.abortRequested():
        if monitor.waitForAbort(2):
            process_trigger_file('install-lbrynet', None)
            process_trigger_file('uninstall-lbrynet', None)
            process_trigger_file('restart-lbrynet', None)
            stop_lbrynet()
            break

        if lbrynet_process:
            if time.time() - time_since_last_cache_clean > clean_cache_interval:
                clean_cache()
                time_since_last_cache_clean = time.time()

            if xbmcaddon.Addon().getSettingBool('predownload_recent_videos'):
                if time.time() - time_since_last_predownload > predownload_interval:
                    predownload()
                    time_since_last_predownload = time.time()

        process_trigger_file('install-lbrynet', install_lbrynet)
        process_trigger_file('uninstall-lbrynet', uninstall_lbrynet)
        process_trigger_file('restart-lbrynet', restart_lbrynet)
